define('feature/changeset/tree-and-diff-view', [
    'jquery',
    'underscore',
    'util/dom-event',
    'util/events',
    'util/feature-detect',
    'util/function',
    'util/navigator',
    'model/conflict',
    'model/file-change',
    'model/file-content-modes',
    'model/page-state',
    'model/path-and-line',
    'feature/changeset/difftree',
    'feature/file-content',
    'exports'
], function(
    $,
    _,
    domEvent,
    events,
    featureDetect,
    fn,
    navigatorUtil,
    Conflict,
    FileChange,
    FileContentModes,
    pageState,
    PathAndLine,
    difftree,
    FileContent,
    exports
) {
    var DiffTree = difftree.DiffTree,
        ROOT = "ROOT";

    var _options;

    //state
    var currentCommitRange,
        currentFileChange,
        currentFilePath,
        currentSearch,
        changingState = false,
        _destroyables = [];

    // components/features/widgets
    var currentDiffTree,
        diffTreesByCommitRangeId = {}, //cache for diff-tree's created for different CommitRanges
        fileContent;

    // Selectors for resizing changesetPanes height and scrollbar & spinner
    var $window = $(window),
        $footer,
        $content,
        $container,
        $spinner,
        windowHeight,
        diffTreeMaxHeight,
        $changesetFileContent,
        $fileTreeContainer,
        $fileTreeWrapper,
        $fileTree,
        $contentView,
        $diffViewToolbar; // boolean for determining if the file tree is stalking or not

    function getFileChangeFromNode($node) {
        var path = getPathFromNode($node),
            srcPath = getSrcPathFromNode($node),
            changeType = getChangeTypeFromNode($node),
            nodeType = getNodeTypeFromNode($node),
            conflict = getConflictFromNode($node),
            executable = getExecutableFromNode($node),
            srcExecutable = getSrcExecutableFromNode($node);

        return new FileChange({
            repository :  pageState.getRepository(),
            commitRange : currentCommitRange,
            srcPath : srcPath.path,
            path : path.path,
            type : changeType,
            nodeType : nodeType,
            line : path.line,
            search : currentSearch,
            conflict : conflict,
            srcExecutable: srcExecutable,
            executable: executable
        });
    }

    function initFileContent($node) {
        return initFileContentFromChange(getFileChangeFromNode($node));
    }

    function initFileContentFromChange(fileChange) {
        if (!fileContent) {
            fileContent = new FileContent($container, "changeset-file-content");
        }

        currentFileChange = fileChange;
        currentFilePath = new PathAndLine(fileChange.getPath(), fileChange.getLine());
        pageState.setFilePath(fileChange.getPath());

        $container.height($container.height());
        //temporarily set the height explicitly to the current height to stop the jump when the diffview is removed.
        //cleaned up in onTreeAndDiffViewSizeChanged
        var scrollTop = $window.scrollTop();

        return fileContent.init(fileChange, null, null, _options).done(function() {
            $changesetFileContent = $('#changeset-file-content');
            // Don't continue if we don't have a file-content area to work with
            if ($changesetFileContent.length === 0) {
                return;
            }
            $diffViewToolbar = $changesetFileContent.find('.file-toolbar');
            $contentView = $changesetFileContent.find('.content-view');

            scrollTop = scrollContentToTop(scrollTop);
            $window.scrollTop(scrollTop);
        });
    }

    function destroyFileContent() {
        var deferred = $.Deferred();
        currentFilePath = null;
        currentFileChange = null;

        if (fileContent) {
            fileContent.destroy();
            fileContent = null;
        }

        $("#changeset-file-content").remove();

        return deferred.resolve();
    }

    function getPathFromNode($node) {
        return new PathAndLine($node.data('path'));
    }

    function getChangeTypeFromNode($node) {
        return $node.data('changeType');
    }

    function getNodeTypeFromNode($node) {
        return $node.data('nodeType');
    }

    function getSrcPathFromNode($node) {
        return new PathAndLine($node.data('srcPath'));
    }

    function getConflictFromNode($node) {
        return $node.data('conflict') && new Conflict($node.data('conflict'));
    }

    function getSrcExecutableFromNode($node) {
        return $node.data('srcExecutable');
    }

    function getExecutableFromNode($node) {
        return $node.data('executable');
    }

    function onTreeAndDiffViewSizeChanged() {
        windowHeight = $window.height();
        diffTreeMaxHeight = windowHeight - $('.diff-tree-toolbar').outerHeight();

        // update diff-tree height
        $fileTreeWrapper.css({'max-height': diffTreeMaxHeight + 'px', 'border-bottom-width': 0 });
    }

    function scrollContentToTop(scrollTop) {
        var diffOffset = $changesetFileContent.offset();
        if (diffOffset) { // Only try to get the offset if we can get it from the element.
            return Math.min(scrollTop, diffOffset.top);
        }
        return scrollTop;
    }

    // Trigger a state change to refresh the file currently shown in the diff view.
    // Use case: diff options have changed and a new representation of the file needs to be shown.
    events.on('stash.feature.fileContent.optionsChanged', function(change) {
        var nonRefreshKeys = ['hideComments', 'hideEdiff'];

        if(!_.contains(nonRefreshKeys, change.key)) {
            initSelectedFileContent();
        }
    });

    // Keep track of the last search to highlight subsequently selected files in the tree
    events.on('internal.stash.feature.diffView.highlightSearch', function(search) {
        currentSearch = search;
    });

    /**
     * Change the state of the view based on whether the selected file is changed and if we have a current diff-tree
     */
    function onStateChange() {
        changingState = true;

        var selectedPath = getPathFromUrl();

        var selectedFileChanged = (Boolean(selectedPath) ^ Boolean(currentFilePath)) || (selectedPath && selectedPath.path.toString() !== currentFilePath.path.toString());

        if (selectedFileChanged && currentDiffTree) {
            currentDiffTree.selectFile(selectedPath.path.getComponents());
            initSelectedFileContent();
        } else if (selectedPath.toString() !== currentFilePath.toString()) {
            // TODO Using events like this to trigger a line change is not ideal, we need a better way to pass 'messages'
            //      via the fileContent into the current view.
            // Only if the line number has changed directly
            events.trigger('internal.stash.feature.diffView.lineChange', null, selectedPath.line);
            // Otherwise the first selected line will not 'select' again
            currentFilePath = selectedPath;
        }
        changingState = false;
    }

    /**
     * Reload the diff viewer
     *
     * Used when the file changes or the diff view is changed (unified v side-by-side)
     */
    function initSelectedFileContent(){
        var $node = currentDiffTree.getSelectedFile();

        if ($node && $node.length > 0) {
            initFileContent($node);
        } else if (currentFileChange) {
            // Fallback to the current file change, even if there is no tree node selected
            // This is to handle the case where there are no search results but the previous file is still selected
            initFileContentFromChange(currentFileChange);
        }
    }

    function updateDiffTree(optSelectedPathComponents) {
        if (!$spinner) {
            $spinner = $("<div class='spinner'/>");
        }
        $spinner.appendTo("#content .file-tree-wrapper").spin("large");
        return currentDiffTree.init(optSelectedPathComponents).always(function() {
            if ($spinner) {
                $spinner.spinStop().remove();
                $spinner = null;
            }
        }).done(function() {
            $fileTree = $('.file-tree');
            diffTreeMaxHeight = windowHeight - $('.diff-tree-toolbar').outerHeight();
            $fileTreeWrapper.css('max-height', diffTreeMaxHeight);
        });
    }

    function getPathFromUrl() {
        return new PathAndLine(window.location.hash.substring(1));
    }

    var toggleDiffTree;
    function initDiffTreeToggle() {
        var $toggle = $(".collapse-file-tree");
        var $changesetFilesContainer = $(".changeset-files");
        var $diffTreeContainer = $(".file-tree-container");
        var collapsed;


        function triggerCollapse() {
            events.trigger('stash.feature.changeset.difftree.collapseAnimationFinished', null, collapsed);
        }

        $diffTreeContainer.on('transitionend', domEvent.filterByTarget($diffTreeContainer, triggerCollapse));

        toggleDiffTree = function(force) {
            var previousCollapsed = $changesetFilesContainer.hasClass('collapsed');
            $changesetFilesContainer.toggleClass('collapsed', force);

            collapsed = $changesetFilesContainer.hasClass('collapsed');
            if (collapsed !== previousCollapsed) {
                events.trigger('stash.feature.changeset.difftree.toggleCollapse', null, collapsed);

                if (!featureDetect.cssTransition()) {
                    triggerCollapse();
                }
            }
        };

        $toggle.on('click', domEvent.preventDefault(toggleDiffTree));
    }

    function initDiffTree() {
        $('.no-changes-placeholder').remove();

        var filePath = currentFilePath ? currentFilePath : getPathFromUrl();
        return updateDiffTree(filePath.path.getComponents()).then(function(diffTree) {
            var $node = diffTree.getSelectedFile();
            if ($node && $node.length) {
                return initFileContent($node);
            } else {
                return destroyFileContent().done(function() {
                    /* Append a placeholder <div> to keep the table-layout so that
                       the diff-tree does not consume the entire page width */
                    $('.changeset-files').append($("<div class='message no-changes-placeholder'></div>").text(AJS.I18n.getText('stash.web.no.changes.to.show')));
                });
            }
        });
    }

    function createDiffTree(_options) {
        return new DiffTree(".file-tree-wrapper", ".diff-tree-toolbar .aui-toolbar2-primary", currentCommitRange, {
            maxChanges : _options.maxChanges,
            hasOtherParents : _options.numberOfParents > 1,
            urlBuilder : _options.changesUrlBuilder,
            searchUrlBuilder : _options.diffUrlBuilder
        });
    }

    exports.updateCommitRange = function(commitRange) {
        if (commitRange.getId() === currentCommitRange.getId()) {
            // bail out if not actually changing the diff.
            return;
        }

        currentCommitRange = commitRange;
        currentDiffTree.reset(); // unbind any event listeners

        if (Object.prototype.hasOwnProperty.call(diffTreesByCommitRangeId, currentCommitRange.getId())){
            // Use cached difftree if it exists.
            currentDiffTree = diffTreesByCommitRangeId[currentCommitRange.getId()];
        } else {
            currentDiffTree = createDiffTree(_options);
            diffTreesByCommitRangeId[currentCommitRange.getId()] = currentDiffTree;
        }

        initDiffTree();
    };

    function onSelectedNodeChanged($node, initializingTree) {
        // Only set the hash if we're here from a user clicking a file name.
        // If it's a popState or a pushState or hashchange, the hash should already be set correctly.
        // If we're initializing a full tree, we want an empty hash.
        // If we're initializing a full tree BECAUSE of a changeState, the hash should still already be set correctly.
        if (!changingState && !initializingTree) {
            window.location.hash = $node ? getPathFromNode($node).toString() : "";
        }
    }

    function onRequestToggleDiffTreeHandler(keys) {
        if (navigatorUtil.isIE() && navigatorUtil.majorVersion() === 9) {
            // IE9 crashes when you type 't'. "IE crash?! That's unpossible!" you cry.
            // But it's true. The browser executes the full call stack, then crashes before
            // debouncedToggleEvent is called. If you set a breakpoint in that call stack though, it
            // often does NOT crash. The crash doesn't occur without the 'collapsed' class being added.
            // But it only occurs when trigger from the keyboard shortcut (and setTimeout hackery doesn't help).
            return;
        }
        (this.execute ? this : AJS.whenIType(keys)).execute(fn.arity(toggleDiffTree, 0));
    }

    function onRequestMoveToNextHandler(keys) {
        (this.execute ? this : AJS.whenIType(keys)).execute(function() {
            currentDiffTree.openNextFile();
        });
    }

    function onRequestMoveToPreviousHandler(keys) {
        (this.execute ? this : AJS.whenIType(keys)).execute(function() {
            currentDiffTree.openPrevFile();
        });
    }

    exports.init = function(commitRange, options) {
        _options = $.extend({}, exports.defaults, options);

        $footer = $("#footer");
        $content = $("#content");
        $container = $content.find(".changeset-files");
        $fileTreeContainer = $('.file-tree-container');
        $fileTreeWrapper = $fileTreeContainer.children('.file-tree-wrapper');
        windowHeight = $window.height();
        $changesetFileContent = $('#changeset-file-content');

        currentCommitRange = commitRange;
        currentDiffTree = createDiffTree(_options);
        diffTreesByCommitRangeId[currentCommitRange.getId()] = currentDiffTree;
        currentFilePath = getPathFromUrl();

        $window.on('hashchange', onStateChange);

        _destroyables.push(events.chain()
            .on("window.resize", onTreeAndDiffViewSizeChanged)
            .on("stash.feature.fileContent.diffViewExpanded", onTreeAndDiffViewSizeChanged)
            .on('stash.feature.changeset.difftree.selectedNodeChanged', onSelectedNodeChanged)
        );

        initDiffTreeToggle();
        initDiffTree();

        _destroyables.push(events.chain()
            .on('stash.keyboard.shortcuts.requestToggleDiffTreeHandler', onRequestToggleDiffTreeHandler)
            .on('stash.keyboard.shortcuts.requestMoveToNextHandler', onRequestMoveToNextHandler)
            .on('stash.keyboard.shortcuts.requestMoveToPreviousHandler', onRequestMoveToPreviousHandler)
        );

        // Always expand the difftree - hence the 'false' here
        _destroyables.push(events.chainWith(currentDiffTree).on('search-focus', _.partial(toggleDiffTree, false)));
    };

    exports.reset = function() {
        if (currentDiffTree) {
            currentDiffTree.reset();
        }

        currentCommitRange = undefined;
        currentDiffTree = undefined;
        diffTreesByCommitRangeId = {};
        currentFilePath = undefined;
        currentSearch = undefined;

        $window.off('hashchange', onStateChange);

        _.invoke(this._destroyables, 'destroy');

        return destroyFileContent();
    };

    exports.defaults = {
        breadcrumbs : true,
        sourceLink : true,
        changeTypeLozenge : true,
        changeModeLozenge : true,
        contentMode : FileContentModes.DIFF,
        toolbarWebFragmentLocationPrimary : null,
        toolbarWebFragmentLocationSecondary : null
    };

    exports.commentMode = FileContent.commentMode;
});
