package com.atlassian.refapp.ctk.oauth;

import com.atlassian.functest.junit.SpringAwareTestCase;
import com.atlassian.oauth.Consumer;
import com.atlassian.oauth.serviceprovider.ServiceProviderConsumerStore;
import com.atlassian.oauth.util.RSAKeys;
import com.atlassian.sal.api.ApplicationProperties;
import org.junit.After;
import org.junit.Test;

import java.net.URI;
import java.security.PublicKey;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.assertFalse;

public class ServiceProviderConsumerStoreTest extends SpringAwareTestCase
{
    private static final String CONSUMER_KEY = "test-consumer";
    private static final String CONSUMER_PUBLIC_KEY = "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAxZDzGUGk6rElyPm0iOua0lWg84nOlhQN1gmTFTIu5WFyQFHZF6OA4HX7xATttQZ6N21yKMakuNdRvEudyN/coUqe89r3Ae+rkEIn4tCxGpJWX205xVF3Cgsn8ICj6dLUFQPiWXouoZ7HG0sPKhCLXXOvUXmekivtyx4bxVFD9Zy4SQ7IHTx0V0pZYGc6r1gF0LqRmGVQDaQSbivigH4mlVwoAO9Tfccf+V00hYuSvntU+B1ZygMw2rAFLezJmnftTxPuehqWu9xS5NVsPsWgBL7LOi3oY8lhzOYjbMKDWM6zUtpOmWJA52cVJW6zwxCxE28/592IARxlJcq14tjwYwIDAQAB";

    private ServiceProviderConsumerStore consumerStore;
    private ApplicationProperties appProp;

    public void setConsumerStore(ServiceProviderConsumerStore consumerStore)
    {
        this.consumerStore = consumerStore;
    }

    public void setAppProp(ApplicationProperties appProp)
    {
        this.appProp = appProp;
    }

    @Test
    public void testServiceProviderConsumerStoreIsAvailable()
    {
        assertNotNull("ServiceProviderConsumerStore must be available to plugins", consumerStore);
    }

    @Test
    public void testServiceProviderConsumerStoreCanStoreConfiguration() throws Exception
    {
        PublicKey publicKey = RSAKeys.fromPemEncodingToPublicKey(CONSUMER_PUBLIC_KEY);

        final Consumer consumer = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .build();
        consumerStore.put(consumer);

        final Consumer savedConsumer = consumerStore.get(CONSUMER_KEY);
        assertEquals(CONSUMER_KEY, savedConsumer.getKey());
        assertEquals("Test Consumer Name", savedConsumer.getName());
        assertEquals(publicKey, savedConsumer.getPublicKey());
        assertEquals("Consumer Description", savedConsumer.getDescription());
        assertEquals(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"), savedConsumer.getCallback());
    }

    @Test
    public void testServiceProviderConsumerStoreCanStoreOAuth2LOConfiguration() throws Exception
    {
        PublicKey publicKey = RSAKeys.fromPemEncodingToPublicKey(CONSUMER_PUBLIC_KEY);

        final Consumer consumer = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .twoLOAllowed(true)
                .executingTwoLOUser("2lo_user")
                .build();
        consumerStore.put(consumer);

        final Consumer savedConsumer = consumerStore.get(CONSUMER_KEY);
        assertTrue(savedConsumer.getTwoLOAllowed());
        assertEquals("2lo_user", savedConsumer.getExecutingTwoLOUser());
    }

    @Test
    public void testDefault2LOParametersAreAppropriateIfLeftUntouched() throws Exception
    {
        PublicKey publicKey = RSAKeys.fromPemEncodingToPublicKey(CONSUMER_PUBLIC_KEY);

        final Consumer consumer = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .build();
        consumerStore.put(consumer);

        final Consumer savedConsumer = consumerStore.get(CONSUMER_KEY);
        assertFalse(savedConsumer.getTwoLOAllowed());
        assertNull(savedConsumer.getExecutingTwoLOUser());
        assertFalse(savedConsumer.getTwoLOImpersonationAllowed());
    }

    @Test
    public void testServiceProviderConsumerStoreCanStoreOAuth2LOImpersonationConfiguration() throws Exception
    {
        PublicKey publicKey = RSAKeys.fromPemEncodingToPublicKey(CONSUMER_PUBLIC_KEY);

        final Consumer consumer = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .twoLOImpersonationAllowed(true)
                .build();
        consumerStore.put(consumer);

        final Consumer savedConsumer = consumerStore.get(CONSUMER_KEY);
        assertTrue(savedConsumer.getTwoLOImpersonationAllowed());
    }

    @Test
    public void testServiceProviderConsumerStoreCanStoreCanHandle2LOConfigurationChanges() throws Exception
    {
        PublicKey publicKey = RSAKeys.fromPemEncodingToPublicKey(CONSUMER_PUBLIC_KEY);

        final Consumer consumer = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .twoLOAllowed(true)
                .executingTwoLOUser("2lo_user")
                .twoLOImpersonationAllowed(true)
                .build();
        consumerStore.put(consumer);

        final Consumer savedConsumer = consumerStore.get(CONSUMER_KEY);
        assertTrue(savedConsumer.getTwoLOAllowed());
        assertTrue(savedConsumer.getTwoLOImpersonationAllowed());

        final Consumer consumer2 = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .twoLOAllowed(false)
                .executingTwoLOUser("2lo_user_changed")
                .twoLOImpersonationAllowed(false)
                .build();
        consumerStore.put(consumer2);

        final Consumer savedConsumer2 = consumerStore.get(CONSUMER_KEY);
        assertEquals(CONSUMER_KEY, savedConsumer2.getKey());
        assertFalse(savedConsumer2.getTwoLOAllowed());
        assertEquals("2lo_user_changed", savedConsumer2.getExecutingTwoLOUser());
        assertFalse(savedConsumer2.getTwoLOImpersonationAllowed());
    }

    @Test
    public void testIfConsumerDoesNotHave3LOFlagDefinedThenItIsConsideredOn() throws Exception
    {
        PublicKey publicKey = RSAKeys.fromPemEncodingToPublicKey(CONSUMER_PUBLIC_KEY);

        final Consumer consumer = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .build();
        consumerStore.put(consumer);

        final Consumer savedConsumer = consumerStore.get(CONSUMER_KEY);
        assertTrue(savedConsumer.getThreeLOAllowed());
    }

    @Test
    public void testServiceProviderConsumerStoreCanStoreOAuth3LOConfiguration() throws Exception
    {
        PublicKey publicKey = RSAKeys.fromPemEncodingToPublicKey(CONSUMER_PUBLIC_KEY);

        final Consumer consumer = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .threeLOAllowed(false)
                .build();
        consumerStore.put(consumer);

        final Consumer savedConsumer = consumerStore.get(CONSUMER_KEY);
        assertFalse(savedConsumer.getThreeLOAllowed());
    }

    @Test
    public void testServiceProviderConsumerStoreCanHandle3LOConfigurationChanges() throws Exception
    {
        PublicKey publicKey = RSAKeys.fromPemEncodingToPublicKey(CONSUMER_PUBLIC_KEY);

        final Consumer consumer = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .threeLOAllowed(false)
                .build();
        consumerStore.put(consumer);

        final Consumer savedConsumer = consumerStore.get(CONSUMER_KEY);
        assertFalse(savedConsumer.getThreeLOAllowed());

        final Consumer consumer2 = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .threeLOAllowed(true)
                .build();
        consumerStore.put(consumer2);

        final Consumer savedConsumer2 = consumerStore.get(CONSUMER_KEY);
        assertTrue(savedConsumer2.getThreeLOAllowed());
    }

    @Test
    public void testServiceProviderConsumerStoreCanStoreConsumerWith2LOEnabledIndependently() throws Exception
    {
        PublicKey publicKey = RSAKeys.fromPemEncodingToPublicKey(CONSUMER_PUBLIC_KEY);

        final Consumer consumer = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .threeLOAllowed(false)
                .twoLOAllowed(true)
                .build();
        consumerStore.put(consumer);

        final Consumer savedConsumer = consumerStore.get(CONSUMER_KEY);
        assertFalse(savedConsumer.getThreeLOAllowed());
        assertTrue(savedConsumer.getTwoLOAllowed());
        assertFalse(savedConsumer.getTwoLOImpersonationAllowed());
    }

    @Test
    public void testServiceProviderConsumerStoreCanStoreConsumerWith2LOWithImpersonationEnabledIndependently() throws Exception
    {
        PublicKey publicKey = RSAKeys.fromPemEncodingToPublicKey(CONSUMER_PUBLIC_KEY);

        final Consumer consumer = Consumer.key(CONSUMER_KEY)
                .name("Test Consumer Name")
                .publicKey(publicKey)
                .description("Consumer Description")
                .callback(URI.create(appProp.getBaseUrl() + "/consumer/oauthcallback1"))
                .threeLOAllowed(false)
                .twoLOImpersonationAllowed(true)
                .build();
        consumerStore.put(consumer);

        final Consumer savedConsumer = consumerStore.get(CONSUMER_KEY);
        assertFalse(savedConsumer.getThreeLOAllowed());
        assertFalse(savedConsumer.getTwoLOAllowed());
        assertTrue(savedConsumer.getTwoLOImpersonationAllowed());
    }

    @After
    public void tearDown()
    {
        consumerStore.remove(CONSUMER_KEY);
    }
}
