package com.atlassian.plugins.whitelist;

import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugins.whitelist.ao.AoWhitelistRule;
import com.atlassian.plugins.whitelist.ao.AoWhitelistRuleDao;
import com.atlassian.plugins.whitelist.events.WhitelistRuleAddedEvent;
import com.atlassian.plugins.whitelist.events.WhitelistRuleChangedEvent;
import com.atlassian.plugins.whitelist.events.WhitelistRuleRemovedEvent;
import com.atlassian.plugins.whitelist.matcher.RegularExpressionMatcher;
import com.atlassian.sal.api.transaction.TransactionCallback;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.google.common.base.Function;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Collections;
import java.util.UUID;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * @since 1.0
 */
class WhitelistManagerImpl implements WhitelistManager
{
    private final AoWhitelistRuleDao aoWhitelistRuleDao;
    private final TransactionTemplate transactionTemplate;
    private final EventPublisher eventPublisher;

    public WhitelistManagerImpl(final AoWhitelistRuleDao aoWhitelistRuleDao, final TransactionTemplate transactionTemplate, final EventPublisher eventPublisher)
    {
        this.aoWhitelistRuleDao = aoWhitelistRuleDao;
        this.transactionTemplate = transactionTemplate;
        this.eventPublisher = eventPublisher;
    }

    @Override
    public WhitelistRule add(final WhitelistRule whitelistRule)
    {
        return Iterables.getOnlyElement(addAll(Collections.singleton(whitelistRule)));
    }

    @Override
    public Collection<WhitelistRule> addAll(final Iterable<WhitelistRule> whitelistRules)
    {
        checkNotNull(whitelistRules, "whitelistRules");
        final Collection<WhitelistRule> addedWhitelistRules = transactionTemplate.execute(new TransactionCallback<Collection<WhitelistRule>>()
        {
            @Override
            public Collection<WhitelistRule> doInTransaction()
            {
                return ImmutableList.copyOf(Iterables.transform(whitelistRules, new Function<WhitelistRule, WhitelistRule>()
                {
                    @Override
                    public WhitelistRule apply(@Nullable final WhitelistRule whitelistRule)
                    {
                        checkNotNull(whitelistRule, "whitelistRule");
                        validate(whitelistRule);
                        return aoWhitelistRuleDao.add(whitelistRule);
                    }
                }));
            }
        });
        for (final WhitelistRule whitelistRule : addedWhitelistRules)
        {
            eventPublisher.publish(new WhitelistRuleAddedEvent(whitelistRule));
        }
        return addedWhitelistRules;
    }

    @Override
    public WhitelistRule update(final WhitelistRule whitelistRule)
    {
        checkNotNull(whitelistRule, "whitelistRule");
        final Integer whitelistRuleId = whitelistRule.getId();
        checkArgument(whitelistRuleId != null, "Cannot update whitelist rule, the given instance has no database id: " + whitelistRule);
        final WhitelistRule updatedWhitelistRule = transactionTemplate.execute(new TransactionCallback<WhitelistRule>()
        {
            @Override
            @SuppressWarnings("ConstantConditions")
            public WhitelistRule doInTransaction()
            {
                final AoWhitelistRule aoWhitelistRuleData = aoWhitelistRuleDao.get(whitelistRuleId);
                validate(whitelistRule);
                aoWhitelistRuleData.setExpression(whitelistRule.getExpression());
                aoWhitelistRuleData.setType(whitelistRule.getType());
                aoWhitelistRuleData.setAllowInbound(whitelistRule.isAllowInbound());
                aoWhitelistRuleData.save();
                return aoWhitelistRuleData;
            }
        });
        eventPublisher.publish(new WhitelistRuleChangedEvent(updatedWhitelistRule));
        return updatedWhitelistRule;
    }

    @Override
    public void remove(final WhitelistRule whitelistRule)
    {
        removeAll(Collections.singleton(whitelistRule));
    }

    @Override
    public void removeAll(final Iterable<WhitelistRule> whitelistRules)
    {
        checkNotNull(whitelistRules, "whitelistRules");
        final Collection<WhitelistRule> removedWhitelistRules = transactionTemplate.execute(new TransactionCallback<Collection<WhitelistRule>>()
        {
            @Override
            public Collection<WhitelistRule> doInTransaction()
            {
                final Iterable<WhitelistRule> transform = Iterables.transform(whitelistRules, new Function<WhitelistRule, WhitelistRule>()
                {
                    @Override
                    @SuppressWarnings("ConstantConditions")
                    public WhitelistRule apply(final WhitelistRule whitelistRule)
                    {
                        checkNotNull(whitelistRule);
                        final Integer id = whitelistRule.getId();
                        checkArgument(id != null, "Cannot remove whitelist rule, the given instance has no database id: " + whitelistRule);
                        final WhitelistRule aoWhitelistRuleData = aoWhitelistRuleDao.get(id);
                        if (aoWhitelistRuleData == null)
                        {
                            return null;
                        }
                        else
                        {
                            aoWhitelistRuleDao.remove(id);
                            return aoWhitelistRuleData;
                        }
                    }
                });
                return ImmutableList.copyOf(Iterables.filter(transform, Predicates.notNull()));
            }
        });
        for (final WhitelistRule removedWhitelistRule : removedWhitelistRules)
        {
            eventPublisher.publish(new WhitelistRuleRemovedEvent(removedWhitelistRule));
        }
    }

    @Override
    public Collection<WhitelistRule> getAll()
    {
        return transactionTemplate.execute(new TransactionCallback<Collection<WhitelistRule>>()
        {
            @Override
            public Collection<WhitelistRule> doInTransaction()
            {
                return ImmutableList.<WhitelistRule>copyOf(aoWhitelistRuleDao.getAll());
            }
        });
    }

    @Nullable
    @Override
    public WhitelistRule get(final int id)
    {
        return transactionTemplate.execute(new TransactionCallback<WhitelistRule>()
        {
            @Override
            public WhitelistRule doInTransaction()
            {
                return aoWhitelistRuleDao.get(id);
            }
        });
    }

    private static void validate(final WhitelistRule whitelistRule)
    {
        // validate the regular expression by attempting to create a RegularExpressionMatcher out of it
        if (whitelistRule.getType() == WhitelistType.REGULAR_EXPRESSION)
        {
            new RegularExpressionMatcher(whitelistRule.getExpression());
        }
        else if (whitelistRule.getType() == WhitelistType.APPLICATION_LINK)
        {
            try
            {
                UUID.fromString(whitelistRule.getExpression());
            }
            catch (IllegalArgumentException e)
            {
                throw new IllegalArgumentException("Expected whitelist rule of type application link to have an expression of a valid UUID: " + whitelistRule);
            }
        }
    }
}
