package com.atlassian.plugin.webresource.assembler.html;

import com.atlassian.plugin.webresource.ResourceUrl;
import com.atlassian.plugin.webresource.assembler.ResourceUrls;
import com.atlassian.webresource.api.UrlMode;
import com.atlassian.webresource.api.assembler.resource.PluginUrlResource;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.List;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;

/**
 * A hackish approach to making the client request resources lazily.
 * <p>
 * This strategy reverts URLs for CSS/JS batches back in to the various requested keys that created them,
 * then uses those to write some &lt;code&gt;WRM.require(requestedKeys)&lt;/code&gt; JavaScript calls.
 * <p>
 * If constructed with a {@link UrlMode}, it may also write prefetch links for the URLs, assuming that's good for performance.
 * <p>
 * This strategy intentionally makes it the client-side's responsibility to request and load all interactive resources.
 * In the long run, it might not be a great idea. However, for the moment, it helps us work around more rigid internals
 * in the WRM.
 */
public class RequireLazilyTagWriter implements HtmlTagFormatter {
    @Nullable
    private UrlMode urlMode;

    public RequireLazilyTagWriter(@Nullable UrlMode urlMode) {
        this.urlMode = urlMode;
    }

    @Nonnull
    @Override
    public String format(@Nonnull ResourceUrls resourceUrls) {
        requireNonNull(resourceUrls, "The resource urls are mandatory for the creation of the require lazily tag");
        final ResourceUrl batchedResources = resourceUrls.getResourceUrl();
        final String prefix = batchedResources.getBatchType().equals(PluginUrlResource.BatchType.CONTEXT) ? "wrc!" : "wr!";
        final List<String> formattedKeys = Arrays.stream(batchedResources.getKey().split(","))
                .map(key -> "\"" + prefix + key + "\"")
                .collect(toList());

        return "<script>" +
                String.format("WRM.requireLazily([%s])", String.join(",", formattedKeys)) +
                "</script>";
    }

    @Override
    public boolean matches(@Nonnull String resourceName) {
        return true;
    }
}
