package com.atlassian.plugin.webresource.cdn.mapper;

import com.atlassian.plugin.webresource.cdn.CDNStrategy;

import javax.annotation.Nonnull;
import java.util.Optional;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * CDNStrategy that supports Web Resource Mapper (WRM) for pre-baked resources.
 * This class uses Strategy Pattern, so a product can just decorate its current strategy and if
 * some resource is not currently mapped it will fall back to the original implementation.
 * It could also be used to compose multiple CDN mappings if required.
 *
 * @see WebResourceMapper
 * @see MemoryLocalWebResourceMapper for a default implementation
 * @since v3.5.0
 */
public class MappedCDNStrategy implements CDNStrategy {
    protected final Optional<CDNStrategy> cdnStrategy;
    protected final WebResourceMapper webResourceMapper;

    public MappedCDNStrategy(
            @Nonnull final Optional<CDNStrategy> cdnStrategy,
            @Nonnull final WebResourceMapper webResourceMapper) {
        checkNotNull(cdnStrategy, "cdnStrategy is null!");
        checkNotNull(webResourceMapper, "webResourceMapper is null!");
        this.webResourceMapper = webResourceMapper;
        this.cdnStrategy = cdnStrategy;
    }

    @Override
    public boolean supportsCdn() {
        return cdnStrategy.map(CDNStrategy::supportsCdn).orElse(true);
    }

    @Override
    public String transformRelativeUrl(@Nonnull final String url) {
        return webResourceMapper.map(url).stream().findFirst()
                .orElseGet(() -> cdnStrategy.map(s -> s.transformRelativeUrl(url)).orElse(url));
    }
}
