package com.atlassian.plugin.webresource.transformer;

import com.atlassian.plugin.webresource.impl.config.Config;
import com.atlassian.plugin.webresource.impl.snapshot.Resource;
import com.atlassian.plugin.webresource.impl.support.Content;
import com.atlassian.plugin.webresource.impl.support.ContentImpl;
import com.atlassian.plugin.webresource.impl.support.Support;
import com.atlassian.sourcemap.SourceMap;

import java.io.InputStream;
import java.io.OutputStream;

import static com.atlassian.plugin.webresource.impl.support.Support.copy;
import static com.atlassian.webresource.spi.CompilerUtil.toInputStream;

/**
 * Minification transformer, it's not a standard transformer and doesn't follow the transformer API.
 *
 * @since v 3.3.1
 */
public class MinificationTransformer {
    public static Content minify(Config config, final Resource resource, final Content content, final String sourceUrl) {
        if (isMinificationEnabledFor(config, resource)) {
            return new ContentImpl(content.getContentType(), true) {
                @Override
                public SourceMap writeTo(OutputStream out, boolean isSourceMapEnabled) {
                    String pathForMinifiedVersion = getPathForMinifiedVersion(resource.getPath());

                    InputStream minifiedStream = null;

                    // Provide Globally minified resource if Global Minification is Enabled
                    if (config.isGlobalMinificationEnabled() && resource.getNameOrLocationType().equals(Config.JS_TYPE)) {
                        minifiedStream = toInputStream(config.getResourceCompiler(), resource.getPath());
                    }

                    if (minifiedStream == null) {
                        minifiedStream = resource.getStreamFor(pathForMinifiedVersion);
                    }

                    // Checking alternate path for minified resource.
                    if (minifiedStream == null) {
                        pathForMinifiedVersion = getAlternatePathForMinifiedVersion(resource.getPath());
                        minifiedStream = resource.getStreamFor(pathForMinifiedVersion);
                    }


                    if (minifiedStream == null) {
                        // If there's no minified version serving original version.
                        content.writeTo(out, isSourceMapEnabled);
                        return null;
                    } else {
                        copy(minifiedStream, out);
                        if (isSourceMapEnabled)
                        {
                            try
                            {
                                return Support.getSourceMap(pathForMinifiedVersion, resource, sourceUrl);
                            }
                            catch (RuntimeException e)
                            {
                                Support.LOGGER.warn("can't parse source map for " + resource.getKey() + "/" + resource.getName(), e);
                                return null;
                            }
                        } else {
                            return null;
                        }
                    }
                }
            };
        } else {
            return content;
        }
    }

    /**
     * If minification enabled for given resource.
     */
    private static boolean isMinificationEnabledFor(Config config, Resource resource) {
        // check if minification has been turned off for this resource (at the
        // module level)
        if (!resource.getParent().isMinificationEnabled()) {
            return false;
        }

        if (!config.isMinificationEnabled()) {
            return false;
        }

        String path = resource.getPath();
        // We only minify .js or .css files
        if (path.endsWith(".js")) {
            // Check if it is already the minified version of the file
            return !(path.endsWith("-min.js") || path.endsWith(".min.js"));
        }
        if (path.endsWith(".css")) {
            // Check if it is already the minified version of the file
            return !(path.endsWith("-min.css") || path.endsWith(".min.css"));
        }
        // Not .js or .css, don't bother trying to find a minified version (may
        // save some file operations)
        return false;
    }

    /**
     * The path to the minified version of the resource.
     */
    private static String getPathForMinifiedVersion(String path) {
        final int lastDot = path.lastIndexOf(".");
        // this can never but -1 since the method call is protected by a call to
        // minificationStrategyInPlay() first
        return path.substring(0, lastDot) + "-min" + path.substring(lastDot);
    }

    /**
     * The path to the minified version of the resource.
     */
    private static String getAlternatePathForMinifiedVersion(String path) {
        final int lastDot = path.lastIndexOf(".");
        // this can never but -1 since the method call is protected by a call to
        // minificationStrategyInPlay() first
        return path.substring(0, lastDot) + ".min" + path.substring(lastDot);
    }
}