package com.atlassian.plugin.webresource.url;

import com.atlassian.plugin.webresource.Globals;
import com.atlassian.plugin.webresource.WebResourceModuleDescriptor;
import com.atlassian.plugin.webresource.condition.ConditionState;
import com.atlassian.plugin.webresource.condition.ConditionsCache;
import com.atlassian.plugin.webresource.transformer.StaticTransformers;
import com.atlassian.plugin.webresource.transformer.TransformerCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;

/**
 * Lazily constructs {@link com.atlassian.plugin.webresource.url.DefaultUrlBuilder} instances for a given type.
 *
 * @since v3.0.5
 */
public class DefaultUrlBuilderMap
{
    private static final Logger log = LoggerFactory.getLogger(DefaultUrlBuilderMap.class);
    private final Iterable<WebResourceModuleDescriptor> wrmds;
    private final StaticTransformers staticTransformers;
    private final Map<String, UrlParameters> urlBuildersByType;
    private final TransformerCache transformerCache;

    public DefaultUrlBuilderMap(WebResourceModuleDescriptor wrmd, TransformerCache transformerCache, StaticTransformers staticTransformers)
    {
        this.staticTransformers = staticTransformers;
        this.wrmds = ImmutableList.of(wrmd);
        this.transformerCache = transformerCache;
        this.urlBuildersByType = Maps.newHashMap();
    }

    public DefaultUrlBuilderMap(Iterable<WebResourceModuleDescriptor> wrmds, TransformerCache transformerCache, StaticTransformers staticTransformers)
    {
        this.staticTransformers = staticTransformers;
        this.wrmds = Lists.newArrayList(wrmds);
        this.transformerCache = transformerCache;
        this.urlBuildersByType = Maps.newHashMap();
    }

    /**
     * Returns a UrlParameters for the given type.
     * If none has been created for the given type, one is initialised by calling
     * {@link com.atlassian.plugin.webresource.WebResourceModuleDescriptor#addToUrl(String, com.atlassian.plugin.webresource.transformer.StaticTransformers, com.atlassian.plugin.webresource.transformer.TransformerCache, DefaultUrlBuilder, com.atlassian.plugin.webresource.condition.ConditionState)}
     * for each WebResourceModuleDescriptor that this url builder is aware of.
     * @param type resource type (eg js / css)
     * @param conditionsRun
     * @return UrlParameters for the given type
     */
    public UrlParameters getOrCreateForType(String type, ConditionState conditionsRun, ConditionsCache conditionsCache, Globals globals)
    {
        UrlParameters urlParams = urlBuildersByType.get(type);
        if (null == urlParams)
        {
            DefaultUrlBuilder urlBuilder = new DefaultUrlBuilder();
            for (WebResourceModuleDescriptor wrmd : wrmds)
            {
                try
                {
                    wrmd.addToUrl(type, staticTransformers, transformerCache, urlBuilder, conditionsRun, conditionsCache);
                }
                catch (RuntimeException e) {
                    // PLUGWEB-261 there's an unknown bug that seems to be caused by reading the non existing resource.
                    // It's unclear what may cause it - the stale cache in WRM or other reasons.
                    // This workaround doesn't fix it but at least it silence the exception and allows other resources
                    // to be served.
                    boolean hasWebResourceInGlobals = globals.getBundles().get(wrmd.getCompleteKey()) != null;
                    boolean hasWebResourceInPluginSystem = globals.getConfig().getIntegration().getPluginAccessor().getEnabledPluginModule(wrmd.getCompleteKey()) != null;
                    log.error(
                        "can't get transformer parameters for \"" + wrmd.getCompleteKey() + "\" (" +
                        "it " + (hasWebResourceInGlobals ? "exists in globals" : "doesn't exists in globals") +
                        ", it " + (hasWebResourceInPluginSystem? "exists in plugin system" : "doesn't exists in plugin system") +
                        ")", e
                    );
                }
            }
            urlParams = urlBuilder.build();
            urlBuildersByType.put(type, urlParams);
        }
        return urlParams;
    }
}
