package com.atlassian.plugin.webresource.assembler;

import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheFactory;
import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.fugue.Option;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.webresource.Globals;
import com.atlassian.plugin.webresource.WebResourceFilter;
import com.atlassian.plugin.webresource.WebResourceModuleDescriptor;
import com.atlassian.plugin.webresource.condition.ConditionState;
import com.atlassian.plugin.webresource.condition.ConditionsCache;
import com.atlassian.plugin.webresource.condition.DecoratingCondition;
import com.atlassian.plugin.webresource.transformer.StaticTransformers;
import com.atlassian.plugin.webresource.transformer.TransformableResource;
import com.atlassian.plugin.webresource.transformer.TransformerCache;
import com.atlassian.plugin.webresource.url.DefaultUrlBuilderMap;
import com.atlassian.plugin.webresource.url.UrlParameters;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.ImmutableSet.copyOf;

/**
* A Cache from requested resources, contexts and (prior) include state to WebResourceSets.
*
* @since 3.1.1
*/
class WebResourceSetCache
{
    private static final Logger log = LoggerFactory.getLogger(WebResourceSetCache.class);
    private static final String RESOURCE_TO_CONDITION_LIST_CACHE_NAME = "resourceToConditionList";
    private static final String RESOURCE_AND_CONDITIONS_TO_SET_CACHE_NAME = "resourceAndConditionsToSet";
    private static final boolean CACHE_ENABLED = Boolean.valueOf(System.getProperty("plugin.webresource.tagcache.enabled", "true"));

    private final Cache<WebResourceSetCacheKey, WebResourceSetConditionEntry> resourceToConditionListCache;
    private final Cache<ResourceAndConditionStateKey, WebResourceSetEntry> resourceAndConditionToSetCache;
    private final StaticTransformers staticTransformers;
    private final Globals globals;

    WebResourceSetCache(Globals globals, CacheFactory cacheFactory, StaticTransformers staticTransformers)
    {
        this.globals = globals;
        this.staticTransformers = staticTransformers;

        CacheSettings resourceToConditionListCacheSettings = new CacheSettingsBuilder()
            .local()
            .build();
        resourceToConditionListCache = getCache(cacheFactory, WebResourceSetCache.class, RESOURCE_TO_CONDITION_LIST_CACHE_NAME, resourceToConditionListCacheSettings);
        CacheSettings resourceAndConditionToSetCacheSettings = new CacheSettingsBuilder()
            .local()
            .build();
        resourceAndConditionToSetCache = getCache(cacheFactory, WebResourceSetCache.class,
            RESOURCE_AND_CONDITIONS_TO_SET_CACHE_NAME, resourceAndConditionToSetCacheSettings);
    }

    private static <K, V> Cache<K, V> getCache(CacheFactory cacheFactory, Class<?> owningClass, String name, CacheSettings required)
    {
        String cacheName = owningClass.getName() + "." + name;
        return cacheFactory.getCache(cacheName, null, required);
    }

    /**
     * Lookup the cache for a WebResourceSet for the given key and context.
     * @param key the key to find
     * @return some WebResourceSet if an entry was found in the cache that matches the conditions run previously. Otherwise, none.
     */
    public Option<WebResourceSetEntry> get(WebResourceSetCacheKey key, TransformerCache transformerCache, ConditionsCache conditionsCache)
    {
        if (!CACHE_ENABLED)
            return Option.none();

        WebResourceSetConditionEntry conditionCache = resourceToConditionListCache.get(key);
        if (conditionCache == null)
        {
            log.debug("No entry for key: {}", key);
            return Option.none();
        }

        // We need to check if the conditions run for the cached entry are still the same. If not, we need to recalculate the set
        ConditionState currentState = getCurrentConditionState(new ConditionState(), conditionCache, conditionsCache);
        Map<String, UrlParameters> parameters = getCurrentParameters(conditionCache, transformerCache, conditionsCache);
        ConditionStateKey stateKey = new ConditionStateKey(currentState, parameters);
        ResourceAndConditionStateKey secondaryKey = new ResourceAndConditionStateKey(key, stateKey);

        WebResourceSetEntry entry = resourceAndConditionToSetCache.get(secondaryKey);
        if (entry != null)
        {
            log.debug("Conditions are fair for key: {}", key);
            return Option.some(entry.copy());
        }
        log.debug("Still no entry for key: {}", key);
        return Option.none();
    }

    public void add(WebResourceSetCacheKey key, ConditionState conditionsRun, InclusionState inclusion
        , DefaultWebResourceSet.Builder set, TransformerCache transformerCache, ConditionsCache conditionsCache)
    {
        if (!CACHE_ENABLED)
            return;

        WebResourceSetConditionEntry conditionCache = resourceToConditionListCache.get(key);
        while (conditionCache == null)
        {
            resourceToConditionListCache.putIfAbsent(key, new WebResourceSetConditionEntry());
            conditionCache = resourceToConditionListCache.get(key);
        }

        synchronized (conditionCache.lock)
        {
            boolean conditionsChanged = conditionCache.allConditionsToRun.addAll(conditionsRun.getConditions());
            boolean resourceTypesChanged = conditionCache.allResourceTypes.addAll(conditionsRun.getResourceTypes());
            boolean modulesChanged = conditionCache.allWebResourceModuleDescriptors.addAll(conditionsRun.getWebResourceModuleDescriptors());
            if (conditionsChanged || resourceTypesChanged || modulesChanged)
            {
                // uhoh, the cache keys need updating to include the extra conditions/transforms
                Collection<ResourceAndConditionStateKey> keys = resourceAndConditionToSetCache.getKeys();
                for (ResourceAndConditionStateKey stateKey : keys)
                {
                    if (stateKey.matches(key))
                        resourceAndConditionToSetCache.remove(stateKey);
                }

            }
            ConditionState allConditionsRun = getCurrentConditionState(conditionsRun, conditionCache, conditionsCache);

            Map<String, UrlParameters> parameters = getCurrentParameters(conditionCache, transformerCache, conditionsCache);
            ConditionStateKey stateKey = new ConditionStateKey(allConditionsRun, parameters);
            ResourceAndConditionStateKey secondaryKey = new ResourceAndConditionStateKey(key, stateKey);
            resourceAndConditionToSetCache.put(secondaryKey, new WebResourceSetEntry(
                conditionsRun.getWebResourceModuleDescriptors(), inclusion, set));
        }
    }

    private ConditionState getCurrentConditionState(ConditionState conditionsRun,
                                                    WebResourceSetConditionEntry conditionCache, ConditionsCache conditionsCache)
    {
        ConditionState allConditionsRun = new ConditionState();
        for (DecoratingCondition condition : conditionCache.allConditionsToRun)
        {
            Boolean value = conditionsRun.getConditionResult(condition);
            if (value == null)
            {
                value = conditionsCache.shouldDisplayImmediate(condition);
            }
            allConditionsRun.addCondition(condition, value);
        }
        return allConditionsRun;
    }

    private Map<String, UrlParameters> getCurrentParameters(WebResourceSetConditionEntry conditionCache,
                                                            TransformerCache transformerCache,
                                                            ConditionsCache conditionsCache)
    {
        Map<String, UrlParameters> params = new HashMap<String, UrlParameters>();
        DefaultUrlBuilderMap urlBuilderMap = new DefaultUrlBuilderMap(conditionCache.allWebResourceModuleDescriptors, transformerCache, staticTransformers);
        Set<String> resourceTypes = conditionCache.allResourceTypes;
        for (String resourceType : resourceTypes)
        {
            params.put(resourceType, getCurrentParameters(urlBuilderMap, resourceType, conditionsCache));
        }
        return params;
    }

    private UrlParameters getCurrentParameters(DefaultUrlBuilderMap urlBuilderMap,
                                               String resourceType,
                                               ConditionsCache conditionsCache)
    {
        ConditionState conditionsRun = new ConditionState();
        return urlBuilderMap.getOrCreateForType(resourceType, conditionsRun, conditionsCache, globals);
    }

    public void clear()
    {
        resourceToConditionListCache.removeAll();
        resourceAndConditionToSetCache.removeAll();
    }

    public int numIncludeVariants()
    {
        return resourceToConditionListCache.getKeys().size();
    }

    public int numEntries()
    {
        return resourceAndConditionToSetCache.getKeys().size();
    }

    public static class ResourceAndConditionStateKey
    {
        private final WebResourceSetCacheKey webResourceSetCacheKey;
        private final ConditionStateKey conditionStateKey;

        public ResourceAndConditionStateKey(WebResourceSetCacheKey webResourceSetCacheKey,
                                            ConditionStateKey conditionStateKey)
        {
            this.webResourceSetCacheKey = webResourceSetCacheKey;
            this.conditionStateKey = conditionStateKey;
        }

        @Override
        public boolean equals(Object o)
        {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            ResourceAndConditionStateKey that = (ResourceAndConditionStateKey) o;

            if (!conditionStateKey.equals(that.conditionStateKey)) return false;
            if (!webResourceSetCacheKey.equals(that.webResourceSetCacheKey)) return false;

            return true;
        }

        @Override
        public int hashCode()
        {
            int result = webResourceSetCacheKey.hashCode();
            result = 31 * result + conditionStateKey.hashCode();
            return result;
        }

        public boolean matches(WebResourceSetCacheKey key)
        {
            return this.webResourceSetCacheKey.equals(key);
        }
    }

    public static class WebResourceSetConditionEntry
    {
        final Object lock = new Object();
        Set<DecoratingCondition> allConditionsToRun = new HashSet<DecoratingCondition>();
        Set<String> allResourceTypes = new HashSet<String>();
        Set<WebResourceModuleDescriptor> allWebResourceModuleDescriptors = new HashSet<WebResourceModuleDescriptor>();
    }

    public static class ConditionStateKey
    {
        private final ConditionState conditionState;
        private final Map<String, UrlParameters> urlParameters;

        public ConditionStateKey(ConditionState conditionState, Map<String, UrlParameters> urlParameters)
        {
            this.conditionState = conditionState;
            this.urlParameters = urlParameters;
        }

        @Override
        public boolean equals(Object o)
        {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            ConditionStateKey that = (ConditionStateKey) o;

            if (!conditionState.equals(that.conditionState)) return false;
            if (!urlParameters.equals(that.urlParameters)) return false;

            return true;
        }

        @Override
        public int hashCode()
        {
            int result = conditionState.hashCode();
            result = 31 * result + urlParameters.hashCode();
            return result;
        }
    }

    public static class WebResourceSetEntry
    {
        private final Iterable<WebResourceModuleDescriptor> includedModules;
        private final InclusionState resultingInclusionState;
        private final DefaultWebResourceSet.Builder builder;

        public WebResourceSetEntry(final Iterable<WebResourceModuleDescriptor> includedModules,
                                   final InclusionState resultingInclusionState,
                                   final DefaultWebResourceSet.Builder builder)
        {
            checkNotNull(builder);
            this.includedModules = includedModules;
            this.resultingInclusionState = resultingInclusionState;
            this.builder = builder.copy();
        }

        public Iterable<WebResourceModuleDescriptor> includedModules()
        {
            return includedModules;
        }

        public InclusionState inclusion()
        {
            return resultingInclusionState;
        }

        public DefaultWebResourceSet.Builder webResourceSet()
        {
            return builder;
        }

        public WebResourceSetEntry copy()
        {
            return new WebResourceSetEntry(this.includedModules, this.resultingInclusionState, this.builder);
        }
    }

    public static class WebResourceSetCacheKey
    {
        private final Set<String> requiredWebResources;
        private final Set<String> requiredContexts;
        private final Set<String> requiredDataKeys;
        private final WebResourceFilter filter;
        private final InclusionState startingInclusionState;

        public WebResourceSetCacheKey(Set<String> requiredWebResources,
                                      Set<String> requiredContexts,
                                      Set<String> requiredDataKeys,
                                      WebResourceFilter filter,
                                      InclusionState startingInclusionState)
        {
            this.requiredWebResources = copyOf(requiredWebResources);
            this.requiredContexts = copyOf(requiredContexts);
            this.requiredDataKeys = copyOf(requiredDataKeys);
            this.filter = filter;
            this.startingInclusionState = startingInclusionState.copy();
        }

        @Override
        public boolean equals(Object o)
        {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            WebResourceSetCacheKey that = (WebResourceSetCacheKey) o;

            if (filter != null ? !filter.equals(that.filter) : that.filter != null) return false;
            if (startingInclusionState != null ? !startingInclusionState.equals(that.startingInclusionState) : that.startingInclusionState != null) return false;
            if (requiredContexts != null ? !requiredContexts.equals(that.requiredContexts) :
                that.requiredContexts != null)
            {
                return false;
            }
            if (requiredDataKeys != null ? !requiredDataKeys.equals(that.requiredDataKeys) : that.requiredDataKeys != null)
                return false;
            if (requiredWebResources != null ? !requiredWebResources.equals(that.requiredWebResources) :
                that.requiredWebResources != null)
            {
                return false;
            }

            return true;
        }

        @Override
        public int hashCode()
        {
            int result = 0;
            result = 31 * result + (requiredWebResources != null ? requiredWebResources.hashCode() : 0);
            result = 31 * result + (requiredContexts != null ? requiredContexts.hashCode() : 0);
            result = 31 * result + (requiredDataKeys != null ? requiredDataKeys.hashCode() : 0);
            result = 31 * result + (filter != null ? filter.hashCode() : 0);
            result = 31 * result + (startingInclusionState != null ? startingInclusionState.hashCode() : 0);
            return result;
        }

        @Override
        public String toString()
        {
            return "WebResourceSetCacheKey{" +
                "requiredWebResources=" + requiredWebResources +
                ", requiredContexts=" + requiredContexts +
                ", requiredDataKeys=" + requiredDataKeys +
                ", filter=" + filter +
                ", inclusion=" + startingInclusionState +
                '}';
        }
    }
}
