package com.atlassian.plugin.webresource;

import com.atlassian.plugin.webresource.http.Router;
import com.atlassian.plugin.webresource.url.UrlParameters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.collect.Iterables.concat;

/**
 * Represents a batch of all resources that declare themselves as part of a given context(s).
 * <p/>
 * The URL for batch resources is /download/contextbatch/&lt;type>/&lt;contextname>/batch.&lt;type. The additional type part in the path
 * is simply there to make the number of path-parts identical with other resources, so relative URLs will still work
 * in CSS files.
 * <p/>
 * The &lt;contextname&gt; part has the following syntax - <br/>
 * &lt;includedContext1&gt;,&lt;includedContext2&gt;,-&lt;excludedContext1&gt;,-&lt;excludedContext2&gt;
 * <br/>
 * To clarify, multiple comma separated contexts should be combined with any contexts preceded by a '-' to be excluded.
 *
 * @since 2.9.0
 */
public class ContextBatchPluginResource extends AbstractPluginResource
{
    private final String key;
    private final String versionHash;
    private final List<String> contexts;
    private final Iterable<String> excludedContexts;
    private final String name;
    private final String type;

    ContextBatchPluginResource(Globals globals, final String key, List<String> contexts, final Iterable<String> excludedContexts,
            final String versionHash, final String type, final UrlParameters urlParams,
            final Map<String, String> params,
            Set<BatchedWebResourceDescriptor> batchedWebResourceDescriptor)
    {
        super(globals, urlParams, params, Collections.unmodifiableSet(batchedWebResourceDescriptor));
        this.name = "batch." + type;
        this.type = type;
        this.key = key;
        this.versionHash = versionHash;
        this.contexts = contexts;
        this.excludedContexts = excludedContexts;
    }

    Iterable<String> getContexts()
    {
        return contexts;
    }

    public Iterable<String> getExcludedContexts()
    {
        return excludedContexts;
    }    

    @Override
    public String getVersion()
    {
        return versionHash;
    }

    @Override
    public String getType()
    {
        return type;
    }

    @Override
    public boolean isCacheSupported()
    {
        return true;
    }

    @Override
    public String getResourceName()
    {
        return name;
    }

    @Override
    public String getCompleteKey()
    {
        return "contextbatch-" + name;
    }

    public String getContext()
    {
        return key;
    }

    public Collection<ContextBatchPluginResource> splitIntoParts() {
        if (contexts.size() <= 1) {
            return Collections.singletonList(this);
        }

        List<ContextBatchPluginResource> result = new LinkedList<ContextBatchPluginResource>();
        List<String> furtherExcludes = new ArrayList<String>();
        for (String ctx : contexts) {
            // just a context batch of 'ctx', excluding the ones we've already sent
            Iterable<String> totalExcludes = concat(excludedContexts, furtherExcludes);
            List<String> newContexts = Collections.singletonList(ctx);
            String newKey = Router.encodeContexts(newContexts, totalExcludes);
            result.add(new ContextBatchPluginResource(globals, newKey, newContexts, totalExcludes, versionHash, getType(), urlParams, params, batchedWebResourceDescriptors));

            furtherExcludes.add(ctx); // now exclude next time
        }
        return result;
    }

    @Override
    public String getUrl(boolean isAbsolute)
    {
        return globals.getRouter().cloneWithNewUrlMode(isAbsolute).urlFor(this);
    }

    @Override
    public Map<String, String> getParamsWithBatchableParams()
    {
        return ResourceUtils.mergeParamList(params, urlParams.toQueryString());
    }

    @Override
    public String toString()
    {
        return "[Context Batch type=" + getType() + ", params=" + params + "]";
    }
}
