package com.atlassian.plugin.webresource;

import com.atlassian.plugin.webresource.url.CDNSupport;
import com.atlassian.plugin.webresource.url.NamedHash;
import com.atlassian.plugin.webresource.url.UrlParameters;

import static com.atlassian.plugin.servlet.AbstractFileServerServlet.PATH_SEPARATOR;
import static com.atlassian.plugin.servlet.AbstractFileServerServlet.SERVLET_PATH;
import static com.google.common.collect.Iterables.concat;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Collection;
import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;

/**
 * Represents a batch of all resources that declare themselves as part of a given context(s).
 * <p/>
 * The URL for batch resources is /download/contextbatch/&lt;type>/&lt;contextname>/batch.&lt;type. The additional type part in the path
 * is simply there to make the number of path-parts identical with other resources, so relative URLs will still work
 * in CSS files.
 * <p/>
 * The &lt;contextname&gt; part has the following syntax - <br/>
 * &lt;includedContext1&gt;,&lt;includedContext2&gt;,-&lt;excludedContext1&gt;,-&lt;excludedContext2&gt;
 * <br/>
 * To clarify, multiple comma separated contexts should be combined with any contexts preceded by a '-' to be excluded.
 *
 * @since 2.9.0
 */
class ContextBatchPluginResource extends AbstractPluginResource
{
    static final String REMOVE_SB_URL_PREFIX = new String(PATH_SEPARATOR + SERVLET_PATH + PATH_SEPARATOR + "contextbatch" + PATH_SEPARATOR);
    static final String INCLUDE_SB_URL_PREFIX = new String(PATH_SEPARATOR + SERVLET_PATH + PATH_SEPARATOR + "contextbatch2" + PATH_SEPARATOR);

    private final ResourceKey resource;
    private final String key;
    private final String versionHash;
    private final List<String> contexts;
    private final Iterable<String> excludedContexts;
    private boolean removeSuperResources;

    ContextBatchPluginResource(final String key, List<String> contexts, final Iterable<String> excludedContexts,
            final String versionHash, final String type, final UrlParameters urlParams,
            final Map<String, String> params,
            Set<BatchedWebResourceDescriptor> batchedWebResourceDescriptor, boolean removeSuperResources)
    {
        super(urlParams, params, Collections.unmodifiableSet(batchedWebResourceDescriptor));
        resource = ResourceKey.Builder.batch(type);
        this.key = key;
        this.versionHash = versionHash;
        this.contexts = contexts;
        this.excludedContexts = excludedContexts;
        this.removeSuperResources = removeSuperResources;
    }

    Iterable<String> getContexts()
    {
        return contexts;
    }

    public Iterable<String> getExcludedContexts()
    {
        return excludedContexts;
    }    

    @Override
    public String getUrl()
    {

        String prefix = removeSuperResources ? REMOVE_SB_URL_PREFIX : INCLUDE_SB_URL_PREFIX;
        final StringBuilder buf = new StringBuilder(prefix.length() + 20);
        buf.append(prefix).append(getType()).append(PATH_SEPARATOR).append(key).append(PATH_SEPARATOR).append(getResourceName());
        ResourceUtils.addParamsToUrl(buf, getParams(), getBatchableParams());
        return buf.toString();
    }

    @Override
    public String getVersion(final WebResourceIntegration integration)
    {
        return versionHash;
    }

    @Override
    public String getType()
    {
        return resource.suffix();
    }

    @Override
    public boolean isCacheSupported()
    {
        return true;
    }

    @Override
    public String getResourceName()
    {
        return resource.name();
    }

    @Override
    public String getModuleCompleteKey()
    {
        return "contextbatch-" + resource.name();
    }

    public Collection<ContextBatchPluginResource> splitIntoParts() {
        if (contexts.size() <= 1) {
            return Collections.singletonList(this);
        }

        List<ContextBatchPluginResource> result = new LinkedList<ContextBatchPluginResource>();
        List<String> furtherExcludes = new ArrayList<String>();
        for (String ctx : contexts) {
            // just a context batch of 'ctx', excluding the ones we've already sent
            Iterable<String> totalExcludes = concat(excludedContexts, furtherExcludes);
            List<String> newContexts = Collections.singletonList(ctx);
            String newKey = ContextBatchOperations.buildContextKey(newContexts, totalExcludes);
            result.add(new ContextBatchPluginResource(newKey, newContexts, totalExcludes, versionHash, getType(), urlParams, params, batchedWebResourceDescriptors, removeSuperResources));

            furtherExcludes.add(ctx); // now exclude next time
        }
        return result;
    }


    @Override
    public String toString()
    {
        return "[Context Batch type=" + getType() + ", params=" + params + "]";
    }
}
