package com.atlassian.plugin.webresource;

import static com.atlassian.plugin.servlet.AbstractFileServerServlet.PATH_SEPARATOR;
import static com.atlassian.plugin.servlet.AbstractFileServerServlet.SERVLET_PATH;

import com.google.common.collect.ImmutableMap;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Creates a batch of all like-typed resources that are declared as "super-batch="true"" in their plugin
 * definitions.
 * <p/>
 * The URL for batch resources is /download/superbatch/&lt;type>/batch.&lt;type. The additional type part in the path
 * is simply there to make the number of path-parts identical with other resources, so relative URLs will still work
 * in CSS files.
 */
public class SuperBatchPluginResource implements PluginResource
{
    static final String URL_PREFIX = PATH_SEPARATOR + SERVLET_PATH + PATH_SEPARATOR + "superbatch" + PATH_SEPARATOR;
    static final String DEFAULT_RESOURCE_NAME_PREFIX = "batch";

    private final ResourceKey resource;
    private final Map<String, String> params;
    private final Set<BatchedWebResourceDescriptor> batchedWebResourceDescriptors;

    public static SuperBatchPluginResource createBatchFor(final PluginResource pluginResource)
    {
        return new SuperBatchPluginResource(pluginResource.getType(), pluginResource.getParams());
    }

    public SuperBatchPluginResource(final String type, final Map<String, String> params)
    {
        this(ResourceKey.Builder.batch(type), params);
    }

    protected SuperBatchPluginResource(final ResourceKey resource, final Map<String, String> params)
    {
        this.resource = resource;
        this.params = ImmutableMap.copyOf(params);
        this.batchedWebResourceDescriptors = new HashSet<BatchedWebResourceDescriptor>();
    }

    public String getUrl()
    {
        final StringBuilder buf = new StringBuilder(URL_PREFIX.length() + 20);
        buf.append(URL_PREFIX).append(getType()).append(PATH_SEPARATOR).append(getResourceName());
        ResourceUtils.addParamsToUrl(buf, getParams());
        return buf.toString();
    }

    public Map<String, String> getParams()
    {
        return params;
    }

    public String getVersion(final WebResourceIntegration integration)
    {
        return integration.getSuperBatchVersion();
    }

    public String getType()
    {
        return resource.suffix();
    }

    public boolean isCacheSupported()
    {
        return true;
    }

    public String getResourceName()
    {
        return resource.name();
    }

    public String getModuleCompleteKey()
    {
        return "superbatch";
    }
    
    void addBatchedWebResourceDescriptor(BatchedWebResourceDescriptor descriptor)
    {
        batchedWebResourceDescriptors.add(descriptor);
    }
    
    @Override
    public Set<BatchedWebResourceDescriptor> getBatchedWebResourceDescriptors()
    {
        return batchedWebResourceDescriptors;
    }

    @Override
    public String toString()
    {
        return "[Superbatch type=" + getType() + ", params=" + getParams() + "]";
    }
}
