package com.atlassian.plugin.webresource.assembler;

import com.atlassian.plugin.webresource.PluginResourceLocator;
import com.atlassian.plugin.webresource.ResourceBatchingConfiguration;
import com.atlassian.plugin.webresource.WebResourceIntegration;
import com.atlassian.plugin.webresource.WebResourceUrlProvider;
import com.atlassian.webresource.api.assembler.WebResourceAssembler;
import com.atlassian.webresource.api.assembler.WebResourceAssemblerBuilder;

import java.util.Collections;
import java.util.List;

/**
 * Implementation of WebResourceAssemblerBuilder
 * @since v3.0
 */
public class DefaultWebResourceAssemblerBuilder implements WebResourceAssemblerBuilder
{
    private final ResourceBatchingConfiguration batchingConfiguration;
    private final WebResourceIntegration webResourceIntegration;
    private final PluginResourceLocator pluginResourceLocator;
    private final WebResourceUrlProvider webResourceUrlProvider;
    private Boolean overrideSuperBatchingEnabled;

    public DefaultWebResourceAssemblerBuilder(ResourceBatchingConfiguration batchingConfiguration,
                                              WebResourceIntegration webResourceIntegration, PluginResourceLocator pluginResourceLocator,
                                              WebResourceUrlProvider webResourceUrlProvider)
    {
        this.batchingConfiguration = batchingConfiguration;
        this.webResourceIntegration = webResourceIntegration;
        this.pluginResourceLocator = pluginResourceLocator;
        this.webResourceUrlProvider = webResourceUrlProvider;
    }

    @Override
    public WebResourceAssembler build()
    {
        ResourceBatchingConfiguration bc =
                overrideSuperBatchingEnabled == null ? batchingConfiguration : new SuperBatchOverridingResourceBatchingConfiguration(batchingConfiguration, overrideSuperBatchingEnabled);
        return new DefaultWebResourceAssembler(bc, webResourceIntegration, pluginResourceLocator,
                webResourceUrlProvider);
    }

    @Override
    public WebResourceAssemblerBuilder includeSuperbatchResources(boolean include) {
        overrideSuperBatchingEnabled = include;
        return this;
    }

    /**
     * Decorating ResourceBatchingConfiguration that overrides superbatch enablement.
     */
    private static class SuperBatchOverridingResourceBatchingConfiguration implements ResourceBatchingConfiguration
    {
        private final ResourceBatchingConfiguration config;
        private final boolean superBatchingEnabled;

        private SuperBatchOverridingResourceBatchingConfiguration(ResourceBatchingConfiguration config, boolean superBatchingEnabled) {
            this.config = config;
            this.superBatchingEnabled = superBatchingEnabled;
        }
        @Override
        public boolean isSuperBatchingEnabled()
        {
            return superBatchingEnabled;
        }

        @Override
        public List<String> getSuperBatchModuleCompleteKeys()
        {
            return superBatchingEnabled ? config.getSuperBatchModuleCompleteKeys() : Collections.<String>emptyList();
        }

        @Override
        public boolean isContextBatchingEnabled()
        {
            return config.isContextBatchingEnabled();
        }

        @Override
        public boolean isPluginWebResourceBatchingEnabled()
        {
            return config.isPluginWebResourceBatchingEnabled();
        }

        @Override
        public boolean isJavaScriptTryCatchWrappingEnabled()
        {
            return config.isJavaScriptTryCatchWrappingEnabled();
        }

        @Override
        public boolean isBatchContentTrackingEnabled()
        {
            return config.isBatchContentTrackingEnabled();
        }

        @Override
        public boolean resplitMergedContextBatchesForThisRequest()
        {
            return config.resplitMergedContextBatchesForThisRequest();
        }
    }
}
