package com.atlassian.plugin.webresource;

import com.google.common.collect.Lists;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Builds superbatch resources
 * @since v3.0
 */
public class SuperBatchBuilder
{
    private final ResourceDependencyResolver dependencyResolver;
    private final PluginResourceLocator pluginResourceLocator;
    private final WebResourceIntegration webResourceIntegration;

    public SuperBatchBuilder(ResourceDependencyResolver dependencyResolver, PluginResourceLocator pluginResourceLocator,
                             WebResourceIntegration webResourceIntegration)
    {
        this.dependencyResolver = dependencyResolver;
        this.pluginResourceLocator = pluginResourceLocator;
        this.webResourceIntegration = webResourceIntegration;
    }

    /**
     * Get all super-batch resources that match the given filter.
     *
     * @return super-batch resources in the order dictated by {@link com.atlassian.plugin.webresource.BatchResourceComparator}. They should be rendered exactly in this order otherwise dependency problems may arise.
     */
    public Iterable<PluginResource> build(WebResourceFilter filter)
    {
        final Iterable<WebResourceModuleDescriptor> superBatchModuleKeys = dependencyResolver.getSuperBatchDependencies();
        final List<PluginResource> resources = new ArrayList<PluginResource>();

        // This is necessarily quite complicated. We need distinct superbatch resources for each combination of
        // resourceFormatter (i.e. separate CSS or JS resources), and also each unique combination of
        // BATCH_PARAMS (i.e. separate superbatches for print stylesheets, IE only stylesheets, and IE only print
        // stylesheets if they ever exist in the future)
        for (final WebResourceFormatter formatter : WebResourceFormatter.webResourceFormatters)
        {
            // This is a map from set of params to batch resource.
            final Map<Map<String,String>, SuperBatchPluginResource> resourceByParams = new HashMap<Map<String,String>,SuperBatchPluginResource>();

            for (final WebResourceModuleDescriptor moduleDescriptor : superBatchModuleKeys)
            {
                for (final PluginResource pluginResource : pluginResourceLocator.getPluginResources(moduleDescriptor.getCompleteKey()))
                {
                    if (formatter.matches(pluginResource.getResourceName()) && filter.matches(pluginResource.getResourceName()))
                    {
                        final Map<String, String> batchParamsMap = getParamMap(pluginResource);
                        // create a new batch resource if the set of param hasn't got a batch resource for it yet.
                        if (!resourceByParams.containsKey(batchParamsMap))
                        {
                            resourceByParams.put(batchParamsMap, SuperBatchPluginResource.createBatchFor(pluginResource));
                        }
                        else
                        {
                            resourceByParams.put(batchParamsMap, SuperBatchOperations.mergeSameBatch(resourceByParams.get(batchParamsMap), pluginResource));
                        }

                        SuperBatchPluginResource superBatchPluginResource = resourceByParams.get(batchParamsMap);

                        // record the content of each super batch to enable front-end dependency tracking
                        BatchedWebResourceDescriptor descriptor = new BatchedWebResourceDescriptor(pluginResource.getVersion(webResourceIntegration), pluginResource.getModuleCompleteKey());
                        superBatchPluginResource.addBatchedWebResourceDescriptor(descriptor);
                    }
                }
            }

            // here we sort the order of the batches. The unconditional batches should come first and the most conditional batches should come last.
            List<SuperBatchPluginResource> resourcesForFormatter = Lists.newArrayList(resourceByParams.values());
            Collections.sort(resourcesForFormatter, BatchResourceComparator.INSTANCE);
            resources.addAll(resourcesForFormatter);
        }

        return resources;
    }

    private Map<String, String> getParamMap(PluginResource pluginResource)
    {
        final Map<String, String> batchParamsMap = new HashMap<String, String>(PluginResourceLocator.BATCH_PARAMS.length);
        for (final String s : PluginResourceLocator.BATCH_PARAMS)
        {
            batchParamsMap.put(s, pluginResource.getParams().get(s));
        }
        return batchParamsMap;
    }
}
