package com.atlassian.plugin.webresource;

import com.atlassian.fugue.Option;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.cache.filecache.FileCache;
import com.atlassian.plugin.cache.filecache.impl.FileCacheImpl;
import com.atlassian.plugin.cache.filecache.impl.PassThroughFileCache;
import com.atlassian.plugin.elements.ResourceDescriptor;
import com.atlassian.plugin.event.PluginEventListener;
import com.atlassian.plugin.event.PluginEventManager;
import com.atlassian.plugin.event.events.PluginDisabledEvent;
import com.atlassian.plugin.event.events.PluginEnabledEvent;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.servlet.ServletContextFactory;
import com.atlassian.plugin.util.PluginUtils;
import com.atlassian.plugin.webresource.cache.FileCacheKey;
import com.atlassian.plugin.webresource.data.DefaultPluginDataResource;
import com.atlassian.plugin.webresource.url.DefaultUrlBuilder;
import com.atlassian.plugin.webresource.url.NamedHash;
import com.atlassian.util.concurrent.LazyReference;
import com.atlassian.webresource.api.data.PluginDataResource;
import com.atlassian.webresource.api.data.WebResourceDataProvider;
import com.google.common.base.Function;
import com.google.common.base.Supplier;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicLong;

import static com.atlassian.plugin.webresource.AbstractBatchResourceBuilder.skipBatch;

/**
 * Default implementation of {@link PluginResourceLocator}.
 *
 * @since 2.2
 */
public class PluginResourceLocatorImpl implements PluginResourceLocator
{
    private static final Logger log = LoggerFactory.getLogger(PluginResourceLocatorImpl.class);

    final private PluginAccessor pluginAccessor;
    final private WebResourceUrlProvider webResourceUrlProvider;
    final private ResourceBatchingConfiguration batchingConfiguration;
    final private List<DownloadableResourceBuilder> builders;
    private final FileCache<FileCacheKey> fileCache;

    /**
     * This field is static to allow multiple instances of {@link FileCacheImpl} in the system to use the same tmp dir,
     * without overwriting each other's files.
     */
    private static final AtomicLong FILENAME_COUNTER = new AtomicLong(0);

    static final String RESOURCE_SOURCE_PARAM = "source";
    static final String RESOURCE_BATCH_PARAM = "batch";
    /** there can easily be 50-80 in a "normal" app, make it 20x bigger */
    static final int DEFAULT_CACHE_SIZE = 1000;

    public PluginResourceLocatorImpl(final WebResourceIntegration webResourceIntegration, final ServletContextFactory servletContextFactory, final WebResourceUrlProvider webResourceUrlProvider, final PluginEventManager pluginEventManager)
    {
        this(webResourceIntegration, servletContextFactory, webResourceUrlProvider, new DefaultResourceBatchingConfiguration(), pluginEventManager);
    }

    public PluginResourceLocatorImpl(final WebResourceIntegration webResourceIntegration,
                                     final ServletContextFactory servletContextFactory,
                                     final WebResourceUrlProvider webResourceUrlProvider,
                                     final ResourceBatchingConfiguration batchingConfiguration, final PluginEventManager pluginEventManager)
    {
        this(webResourceIntegration, servletContextFactory, webResourceUrlProvider, new DefaultResourceDependencyResolver(webResourceIntegration, batchingConfiguration), batchingConfiguration, pluginEventManager);
    }

    private PluginResourceLocatorImpl(final WebResourceIntegration webResourceIntegration, final ServletContextFactory servletContextFactory, final WebResourceUrlProvider webResourceUrlProvider,
                                      final ResourceDependencyResolver dependencyResolver, final ResourceBatchingConfiguration batchingConfiguration,
                                      final PluginEventManager pluginEventManager)
    {
        this.pluginAccessor = webResourceIntegration.getPluginAccessor();
        this.webResourceUrlProvider = webResourceUrlProvider;
        this.batchingConfiguration = batchingConfiguration;
        final SingleDownloadableResourceBuilder singlePluginBuilder =
                new SingleDownloadableResourceBuilder(pluginAccessor, servletContextFactory, webResourceIntegration);

        FileCache<FileCacheKey> cache = makePhysicalCache(webResourceIntegration);
        if (cache == null) {
            cache = PassThroughFileCache.instance();
        }

        this.fileCache = cache;

        builders = Collections.unmodifiableList(Arrays.asList(
                new SuperBatchDownloadableResourceBuilder(dependencyResolver, pluginAccessor, webResourceUrlProvider, singlePluginBuilder, cache, batchingConfiguration),
                new SuperBatchSubResourceBuilder(dependencyResolver, singlePluginBuilder),
                new ContextBatchDownloadableResourceBuilder(dependencyResolver, pluginAccessor, webResourceUrlProvider, singlePluginBuilder, cache, batchingConfiguration),
                new ContextBatchSubResourceBuilder(dependencyResolver, singlePluginBuilder),
                new SingleBatchDownloadableResourceBuilder(pluginAccessor, webResourceUrlProvider, singlePluginBuilder, cache, batchingConfiguration),
                singlePluginBuilder));

        pluginEventManager.register(this);
    }

    private static FileCache<FileCacheKey> makePhysicalCache(WebResourceIntegration webResourceIntegration) {

        boolean cacheDisabled = Boolean.getBoolean(PluginUtils.WEBRESOURCE_DISABLE_FILE_CACHE) || Boolean.getBoolean(PluginUtils.ATLASSIAN_DEV_MODE);
        if (cacheDisabled) {
            return null;
        }


        FileCache<FileCacheKey> cache = null;
        try {
            int cachesize = Integer.getInteger(PluginUtils.WEBRESOURCE_FILE_CACHE_SIZE, DEFAULT_CACHE_SIZE);
            final File tmpDir = webResourceIntegration.getTemporaryDirectory();
            if (tmpDir != null) {
                cache = new FileCacheImpl<FileCacheKey>(tmpDir, cachesize, FILENAME_COUNTER);
            }
        } catch (IOException e) {
            log.error("Could not create file cache object, will startup with filecaching disabled, please investigate the cause and correct it.", e);
        }
        return cache;
    }

    @PluginEventListener
    public void onPluginDisabled(final PluginDisabledEvent event)
    {
        fileCache.clear();
    }

    @PluginEventListener
    public void onPluginEnabled(final PluginEnabledEvent event)
    {
        fileCache.clear();
    }

    public boolean matches(final String url)
    {
        for (final DownloadableResourceBuilder builder : builders)
        {
            if (builder.matches(url))
            {
                return true;
            }
        }

        return false;
    }

    public DownloadableResource getDownloadableResource(final String url, final Map<String, String> queryParams)
    {
        try
        {
            for (final DownloadableResourceBuilder builder : builders)
            {
                if (builder.matches(url))
                {
                    return builder.parse(url, queryParams);
                }
            }
        }
        catch (final UrlParseException e)
        {
            log.error("Unable to parse URL: " + url, e);
        }
        // TODO: It would be better to use Exceptions rather than returning
        // nulls to indicate an error.
        return null;
    }

    @Override
    public Iterable<PluginDataResource> getPluginDataResources(final String moduleCompleteKey)
    {
        Option<WebResourceModuleDescriptor> option = getDescriptor(moduleCompleteKey);
        if (option.isEmpty())
        {
            return Collections.emptyList();
        }

        return Iterables.transform(option.get().getDataProviders().entrySet(), new Function<Map.Entry<String, WebResourceDataProvider>, PluginDataResource>()
        {
            @Override
            public PluginDataResource apply(@Nullable Map.Entry<String, WebResourceDataProvider> input)
            {
                return new DefaultPluginDataResource(moduleCompleteKey + "." + input.getKey(), input.getValue().get());
            }
        });
    }

    @Override
    public List<PluginResource> getPluginResources(final String moduleCompleteKey)
    {
        Option<WebResourceModuleDescriptor> option = getDescriptor(moduleCompleteKey);
        if (option.isEmpty())
        {
            return Collections.emptyList();
        }

        WebResourceModuleDescriptor wrmd = option.get();
        final boolean singleMode = !batchingConfiguration.isPluginWebResourceBatchingEnabled();
        final Set<PluginResource> resources = new LinkedHashSet<PluginResource>();

        DefaultUrlBuilder urlBuilder = new DefaultUrlBuilder();
        wrmd.addToUrl(pluginAccessor, urlBuilder);

        for (final ResourceDescriptor resourceDescriptor : wrmd.getResourceDescriptors())
        {
            if (singleMode || skipBatch(resourceDescriptor))
            {
                final boolean cache = !"false".equalsIgnoreCase(resourceDescriptor.getParameter("cache"));
                resources.add(new SinglePluginResource(resourceDescriptor.getName(), wrmd.getCompleteKey(),
                    cache, urlBuilder.allHashes(),
                    resourceDescriptor.getParameters(), urlBuilder.toQueryString()));
            }
            else
            {
                final BatchPluginResource batchResource = createBatchResource(wrmd.getCompleteKey(), resourceDescriptor,
                        urlBuilder.allHashes(), urlBuilder.toQueryString(),
                        new BatchedWebResourceDescriptor(wrmd.getPlugin().getPluginInformation().getVersion(), wrmd.getCompleteKey()));
                resources.add(batchResource);
            }
        }
        return ImmutableList.copyOf(resources);
    }

    private Option<WebResourceModuleDescriptor> getDescriptor(String moduleCompleteKey)
    {
        final ModuleDescriptor<?> moduleDescriptor = pluginAccessor.getEnabledPluginModule(moduleCompleteKey);
        if ((moduleDescriptor == null) || !(moduleDescriptor instanceof WebResourceModuleDescriptor))
        {
            log.error("Error loading resource \"{}\". Resource is not a Web Resource Module", moduleCompleteKey);
            return Option.none();
        }
        return Option.some((WebResourceModuleDescriptor) moduleDescriptor);
    }

    // package protected so we can test it
    String[] splitLastPathPart(final String resourcePath)
    {
        int indexOfSlash = resourcePath.lastIndexOf('/');
        if (resourcePath.endsWith("/")) // skip over the trailing slash
        {
            indexOfSlash = resourcePath.lastIndexOf('/', indexOfSlash - 1);
        }

        if (indexOfSlash < 0)
        {
            return null;
        }

        return new String[] {resourcePath.substring(0, indexOfSlash + 1), resourcePath.substring(indexOfSlash + 1)};
    }

    private BatchPluginResource createBatchResource(final String moduleCompleteKey,
                                                    final ResourceDescriptor resourceDescriptor,
                                                    final List<NamedHash> hashes,
                                                    final Map<String, String> batchableParams,
                                                    final BatchedWebResourceDescriptor batchedWebResourceDescriptor)
    {
        final Map<String, String> params = new TreeMap<String, String>();
        for (final String param : BATCH_PARAMS)
        {
            final String value = resourceDescriptor.getParameter(param);
            if (StringUtils.isNotEmpty(value))
            {
                params.put(param, value);
            }
        }

        final String name = resourceDescriptor.getName();
        final Supplier<String> type = new LazyReference<String>()
        {
            @Override
            protected String create()
            {
                return name.substring(name.lastIndexOf(".") + 1);
            }
        };

        return new BatchPluginResource(ResourceKey.Builder.lazy(moduleCompleteKey, type), hashes, params, batchableParams, batchedWebResourceDescriptor);
    }

    public String getResourceUrl(final String moduleCompleteKey, final String resourceName)
    {
        return webResourceUrlProvider.getResourceUrl(moduleCompleteKey, resourceName);
    }
}
