package com.atlassian.plugin.webresource;

import com.atlassian.plugin.servlet.DownloadException;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.google.common.base.Function;
import org.apache.commons.lang.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;

import static com.atlassian.plugin.webresource.transformer.TransformerUtils.transformAndStreamResource;

public abstract class AbstractTransformResource implements DownloadableResource
{
    static final Charset UTF8 = Charset.forName("UTF-8");

    protected final DownloadableResource originalResource;

    AbstractTransformResource(final DownloadableResource originalResource)
    {
        this.originalResource = originalResource;
    }

    /**
     * Transform the content.
     *
     * @param originalContent original content
     * @return transformed content.
     */
    abstract CharSequence transform(final CharSequence originalContent);

    public boolean isResourceModified(final HttpServletRequest request, final HttpServletResponse response)
    {
        return originalResource.isResourceModified(request, response);
    }

    public void serveResource(final HttpServletRequest request, final HttpServletResponse response) throws DownloadException
    {
        final String contentType = getContentType();
        if (StringUtils.isNotBlank(contentType))
        {
            response.setContentType(contentType);
        }

        try
        {
            streamResource(response.getOutputStream());
        }
        catch (final IOException e)
        {
            throw new DownloadException(e);
        }
    }

    public void streamResource(final OutputStream out) throws DownloadException
    {
        transformAndStreamResource(originalResource, UTF8, out, new Function<CharSequence, CharSequence>()
        {
            public CharSequence apply(final CharSequence originalContent)
            {
                return transform(originalContent);
            }
        });
    }

    public String getContentType()
    {
        return originalResource.getContentType();
    }

    DownloadableResource getOriginalResource()
    {
        return originalResource;
    }
}