package com.atlassian.webresource.api.prebake;

import com.atlassian.annotations.ExperimentalApi;

import java.util.Collection;
import java.util.stream.Stream;

/**
 * Represents all the possible "dimensions" or set of possible combinations of query parameters
 * that a transformer or condition can support. For example, an i18n transformer may support a
 * "locale" query param that can have values "en", "fr", etc. A condition may support a query
 * param "isadmin" that may be "true" or absent. Both those Dimensions can be combined
 * via {@link #product(Dimensions)} to produce all possible valid combinations of "locale" and "isadmin".
 * <p>
 * For example:
 * </p>
 * <pre>{@code
 * Dimensions locale = Dimensions.empty().andExactly("locale", "en", "fr);
 * Dimensions isadmin = Dimensions.empty().andExactly("isadmin", "true").andAbsent("isadmin");
 * Dimensions dims = locale.product(isadmin);
 * }</pre>
 * <p>
 * This Dimensions concept does not support a query param having multiple values simultaneously. For example,
 * you could not construct a Dimensions that represented <code>?locale=en&amp;locale=fr</code>
 * </p>
 * <p>
 * Instances of this class are immutable, and all methods return results and do not modify the current instance.
 * </p>
 * <p>
 * Implementors of this interface will define equals/hashCode.
 * </p>
 */
@ExperimentalApi
public interface Dimensions
{

    /**
     * Factory for an empty dimension object.
     */
    static Dimensions empty() {
        return DimensionsImpl.empty();
    }

    /**
     * Adds dimensions for query param {@code key} over all the given {@code values}.
     * Does not include the absence of {@code key}.
     */
    Dimensions andExactly(String key, String... values);

    /**
     * Adds dimensions for query param {@code key} over all the given {@code values}.
     * Does not include the absence of {@code key}.
     */
    Dimensions andExactly(String key, Collection<String> values);

    /**
     * Adds a dimension that represents the absence of {@code key} in the query string.
     */
    Dimensions andAbsent(String key);

    /**
     * Combines two dimensions, representing all possible combinations of both dimensions.
     */
    Dimensions product(Dimensions rhs);

    /**
     * Computes the cartesian product as an unordered stream of Coordinates.
     * A Coordinate represents a single possible query-string combination.
     */
    Stream<Coordinate> cartesianProduct();

}
