package com.atlassian.plugin.osgi.factory;

import com.atlassian.plugin.Application;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.ModuleDescriptorFactory;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.PluginArtifact;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.factories.AbstractPluginFactory;
import com.atlassian.plugin.impl.UnloadablePlugin;
import com.atlassian.plugin.osgi.container.OsgiContainerManager;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.io.IOUtils;
import org.dom4j.Element;
import org.osgi.framework.Constants;
import org.osgi.util.tracker.ServiceTracker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.util.jar.Manifest;

import static com.atlassian.plugin.osgi.util.OsgiHeaderUtil.getManifest;
import static com.atlassian.plugin.osgi.util.OsgiHeaderUtil.getPluginKey;
import static com.atlassian.plugin.parsers.XmlDescriptorParserUtils.addModule;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Plugin deployer that deploys OSGi bundles that don't contain XML descriptor files
 */
public final class OsgiBundleFactory extends AbstractPluginFactory
{
    private static final Logger log = LoggerFactory.getLogger(OsgiBundleFactory.class);

    private final OsgiContainerManager osgiContainerManager;
    private final String pluginDescriptorFileName;

    private final OsgiChainedModuleDescriptorFactoryCreator osgiChainedModuleDescriptorFactoryCreator;

    public OsgiBundleFactory(OsgiContainerManager osgi)
    {
        this(PluginAccessor.Descriptor.FILENAME, osgi);
    }

    public OsgiBundleFactory(String pluginDescriptorFileName, OsgiContainerManager osgi)
    {
        super(new OsgiPluginXmlDescriptorParserFactory(), ImmutableSet.<Application>of());
        this.pluginDescriptorFileName = checkNotNull(pluginDescriptorFileName);
        this.osgiContainerManager = checkNotNull(osgi, "The osgi container is required");

        this.osgiChainedModuleDescriptorFactoryCreator = new OsgiChainedModuleDescriptorFactoryCreator(new OsgiChainedModuleDescriptorFactoryCreator.ServiceTrackerFactory()
        {
            public ServiceTracker create(final String className)
            {
                return osgiContainerManager.getServiceTracker(className);
            }
        });
    }

    @Override
    protected InputStream getDescriptorInputStream(PluginArtifact pluginArtifact)
    {
        return pluginArtifact.getResourceAsStream(pluginDescriptorFileName);
    }

    @Override
    protected Predicate<Integer> isValidPluginsVersion()
    {
        return new Predicate<Integer>()
        {
            @Override
            public boolean apply(final Integer input)
            {
                return input != null && input >= Plugin.VERSION_2;
            }
        };
    }

    public String canCreate(final PluginArtifact pluginArtifact) throws PluginParseException
    {
        checkNotNull(pluginArtifact, "The plugin artifact is required");

        InputStream descriptorStream = null;

        try
        {
            descriptorStream = pluginArtifact.getResourceAsStream(pluginDescriptorFileName);
            if (null == descriptorStream)
            {
                final Manifest manifest = getManifest(pluginArtifact);
                if ((null != manifest) && (null != manifest.getMainAttributes().getValue(Constants.BUNDLE_SYMBOLICNAME)))
                {
                    return getPluginKey(manifest);
                }
                else
                {
                    // It's not a bundle - it has no manifest, or the manifest doesn't contain a required key
                    return null;
                }
            }
            else
            {
                // It's an Atlassian Plugin, not a plain bundle
                return null;
            }
        }
        finally
        {
            IOUtils.closeQuietly(descriptorStream);
        }
    }

    /**
     * Create a plugin from the given artifact.
     *
     * @param pluginArtifact the plugin artifact containing the plugin.
     * @param moduleDescriptorFactory The factory for plugin modules.
     * @return The instantiated and populated plugin, or an {@link UnloadablePlugin} if the plugin cannot be loaded.
     * @since 2.2.0
     */
    public Plugin create(final PluginArtifact pluginArtifact, final ModuleDescriptorFactory moduleDescriptorFactory)
    {
        checkNotNull(pluginArtifact, "The plugin artifact is required");
        checkNotNull(moduleDescriptorFactory, "The module descriptor factory is required");

        final String pluginKey = canCreate(pluginArtifact);
        if (null == pluginKey)
        {
            log.warn("Unable to load plugin from '{}'", pluginArtifact);
            return new UnloadablePlugin("PluginArtifact has no manifest or is not a bundle: '" + pluginArtifact + "'");
        }
        else
        {
            return new OsgiBundlePlugin(osgiContainerManager, pluginKey, pluginArtifact);
        }
    }

    @Override
    public ModuleDescriptor<?> createModule(final Plugin plugin, final Element module, final ModuleDescriptorFactory moduleDescriptorFactory)
    {
        if (plugin instanceof OsgiBundlePlugin)
        {
            final ModuleDescriptorFactory combinedFactory = osgiChainedModuleDescriptorFactoryCreator.create(new OsgiChainedModuleDescriptorFactoryCreator.ResourceLocator()
            {
                @Override
                public boolean doesResourceExist(final String name)
                {
                    // This returns true to indicate that the listable module descriptor class is present in this plugin.
                    // Those module descriptors are skipped when first installing the plugin, however no need for us to skip
                    // here, as the plugin has been loaded/installed.
                    return false;
                }
            }, moduleDescriptorFactory);

            return addModule(combinedFactory, plugin, module);
        }
        else
        {
            return null;
        }
    }
}
