package com.atlassian.plugins.osgi.javaconfig.conditions.product;

import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Condition;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.core.type.AnnotatedTypeMetadata;

import static java.util.Objects.requireNonNull;

/**
 * Base class for a Spring bean {@link Condition} that evaluates to <code>true</code> when the plugin is
 * running in a specific host product. It's abstract because condition classes must have a no-arg constructor.
 */
public abstract class AbstractProductCondition implements Condition {

    // Non-static because we want to log as the concrete subclass
    private final Logger logger = LoggerFactory.getLogger(getClass());

    private final String canaryServiceClassName;

    /**
     * Constructor.
     *
     * @param canaryServiceClassName the class name of a service whose presence indicates that we are running a
     *                               specific product
     */
    @SuppressWarnings("WeakerAccess")
    protected AbstractProductCondition(final String canaryServiceClassName) {
        this.canaryServiceClassName = requireNonNull(canaryServiceClassName);
    }

    /*
        This implementation works by testing for the presence of a specific OSGi service that's only available from the
        product in question. This approach is copied from Spring Scanner's ProductFilterUtil class. The original
        implementation called ApplicationProperties#getPlatformId(), however the ApplicationProperties service, being
        provided by the SAL plugin, isn't guaranteed to be available when this condition is being evaluated, so we're
        using this approach instead even though it's less robust (e.g. when the products move/rename/remove the service
        that we're using as a canary).
     */
    @Override
    public final boolean matches(final ConditionContext conditionContext, final AnnotatedTypeMetadata metadata) {
        // This could all be done with Optionals and .map(), but this style allows logging and breakpoints
        final Bundle bundle = FrameworkUtil.getBundle(getClass());
        if (bundle == null) {
            logger.debug("Bundle not found for {}", getClass().getName());
            return false;
        }

        final BundleContext bundleContext = bundle.getBundleContext();
        if (bundleContext == null) {
            logger.debug("BundleContext not found for {}", bundle);
            return false;
        }

        return isCanaryServicePresent(bundleContext);
    }

    /**
     * Indicates whether the canary service exists in the OSGi Service Registry.
     *
     * @param bundleContext the bundle context for obtaining and releasing the service reference
     * @return see above
     */
    private boolean isCanaryServicePresent(final BundleContext bundleContext) {
        ServiceReference canaryServiceReference = null;
        try {
            canaryServiceReference = bundleContext.getServiceReference(canaryServiceClassName);
            final boolean servicePresent = canaryServiceReference != null;
            logger.debug("{} present = {}", canaryServiceClassName, servicePresent);
            return servicePresent;
        } finally {
            if (canaryServiceReference != null) {
                bundleContext.ungetService(canaryServiceReference);
            }
        }
    }
}
