package com.atlassian.plugins.osgi.javaconfig;

import com.atlassian.annotations.PublicApi;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static java.util.Arrays.stream;
import static java.util.Objects.requireNonNull;

/**
 * The user-provided options for exporting a local Spring bean as an OSGi Service.
 *
 * @since 0.3
 */
@PublicApi
public class ExportOptions {

    /**
     * Factory method for an instance that exports a bean under the given interface.
     *
     * @param serviceInterface the interface under which to export the service
     * @return a new instance
     */
    public static ExportOptions as(final Class<?> serviceInterface) {
        return new ExportOptions(serviceInterface);
    }

    private final List<Class<?>> serviceInterfaces = new ArrayList<>();
    private final Map<String, Object> properties = new HashMap<>();

    private ExportOptions(final Class<?> serviceInterface) {
        addServiceInterface(serviceInterface);
    }

    /**
     * Adds further service interfaces under which the bean is to be exported to OSGi.
     *
     * @param otherInterfaces the other interfaces
     * @return this instance
     */
    public ExportOptions andAs(final Class<?>... otherInterfaces) {
        stream(otherInterfaces).forEach(this::addServiceInterface);
        return this;
    }

    private void addServiceInterface(final Class<?> serviceInterface) {
        requireNonNull(serviceInterface);
        if (!serviceInterface.isInterface() || serviceInterface.isEnum()) {
            throw new IllegalArgumentException(serviceInterface.getName() + " is not an interface");
        }
        this.serviceInterfaces.add(serviceInterface);
    }

    /**
     * Adds the given property to this service's map of OSGi properties. Other bundles can filter on these properties.
     *
     * @param key the name of the property
     * @param value the value of the property
     * @return this instance
     */
    public ExportOptions withProperty(final String key, final Object value) {
        properties.put(key, value);
        return this;
    }

    /**
     * Returns the interfaces under which this service is to be exported to OSGi.
     *
     * @return a non-null array
     */
    public Class<?>[] getInterfaces() {
        return new ArrayList<>(serviceInterfaces).toArray(new Class<?>[0]);
    }

    /**
     * Returns the properties to be exported for this service.
     *
     * @return a non-null map
     */
    public Map<String, Object> getProperties() {
        return new HashMap<>(properties);
    }
}
