package com.atlassian.plugin;

import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Multimap;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import static com.atlassian.plugin.PluginDependencies.Type.DYNAMIC;
import static com.atlassian.plugin.PluginDependencies.Type.MANDATORY;
import static com.atlassian.plugin.PluginDependencies.Type.OPTIONAL;
import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableMap;
import static java.util.Collections.unmodifiableSet;
import static java.util.Comparator.comparingInt;

/**
 * All plugin keys which are dependents of another plugin, divided into OSGi style import resolutions: mandatory
 * (default), optional and dynamic.
 *
 * @see Plugin#getDependencies()
 * @since 4.0
 */
public class PluginDependencies {
    // dependency types in the order of significance
    public enum Type {
        MANDATORY,
        OPTIONAL,
        DYNAMIC;

        public boolean lessSignificant(final Type other) {
            return ordinal() > other.ordinal();
        }
    }

    // Plugin keys of mandatory, optional and dynamic dependencies
    // keys can appear in multiple sets
    private final Set<String> mandatory;
    private final Set<String> optional;
    private final Set<String> dynamic;
    /**
     * Plugin keys of all dependencies
     */
    private final Set<String> all;

    private final Multimap<String, Type> byPluginKey;


    public PluginDependencies() {
        this(null, null, null);
    }

    public PluginDependencies(final Set<String> mandatory, final Set<String> optional, final Set<String> dynamic) {
        this.mandatory = mandatory == null ? emptySet() : unmodifiableSet(mandatory);
        this.optional = optional == null ? emptySet() : unmodifiableSet(optional);
        this.dynamic = dynamic == null ? emptySet() : unmodifiableSet(dynamic);

        final Set<String> combined = new HashSet<>();
        combined.addAll(this.mandatory);
        combined.addAll(this.optional);
        combined.addAll(this.dynamic);
        this.all = unmodifiableSet(combined);

        final ImmutableListMultimap.Builder<String, Type> byPluginKeyBuilder = ImmutableListMultimap.builder();

        for (final String key : this.mandatory) {
            byPluginKeyBuilder.put(key, MANDATORY);
        }
        for (final String key : this.optional) {
            byPluginKeyBuilder.put(key, OPTIONAL);
        }
        for (final String key : this.dynamic) {
            byPluginKeyBuilder.put(key, DYNAMIC);
        }

        this.byPluginKey = byPluginKeyBuilder.build();
    }

    public Set<String> getMandatory() {
        return mandatory;
    }

    public Set<String> getOptional() {
        return optional;
    }

    public Set<String> getDynamic() {
        return dynamic;
    }

    public Set<String> getAll() {
        return all;
    }

    /**
     * @deprecated since 5.6.0, this will be removed in the next major version of Atlassian Plugins. See {@link #getTypesByPluginKey()} as an alternative to this.
     * @return guava multimap of plugin key to types.
     */
    @Deprecated
    public Multimap<String, Type> getByPluginKey() {
        return byPluginKey;
    }

    /**
     * @since 5.6.0
     * @return map of pluginKey to supported Types.
     */
    public Map<String, SortedSet<Type>> getTypesByPluginKey() {
        final Map<String, TreeSet<Type>> typesByPluginKeyLocal = new HashMap<>();
        for (Map.Entry<String, Type> entry : byPluginKey.entries()) {
            typesByPluginKeyLocal.computeIfAbsent(entry.getKey(), x -> new TreeSet<>(comparingInt(Enum::ordinal)))
                    .add(entry.getValue());
        }
        return unmodifiableMap(typesByPluginKeyLocal);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        final Set<String> mandatory = new HashSet<>();
        final Set<String> optional = new HashSet<>();
        final Set<String> dynamic = new HashSet<>();

        private Builder() {
        }

        public Builder withMandatory(String... pluginKey) {
            mandatory.addAll(asList(pluginKey));
            return this;
        }

        public Builder withOptional(String... pluginKey) {
            optional.addAll(asList(pluginKey));
            return this;
        }

        public Builder withDynamic(String... pluginKey) {
            dynamic.addAll(asList(pluginKey));
            return this;
        }

        public PluginDependencies build() {
            return new PluginDependencies(unmodifiableSet(mandatory), unmodifiableSet(optional), unmodifiableSet(dynamic));
        }
    }

}
