package com.atlassian.depview.rest;

import com.atlassian.depview.osgi.BundleExplorer;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.osgi.util.OsgiHeaderUtil;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

import javax.annotation.Nonnull;
import javax.inject.Inject;
import javax.inject.Singleton;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.Arrays;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.atlassian.depview.osgi.WireDirection.PROVIDED;
import static com.atlassian.depview.osgi.WireDirection.REQUIRED;

@Singleton
@Path("/bundles")
public class BundlesResource {
    private final BundleContext bundleContext;

    private final UriGenerator uriGenerator;

    private final PluginAccessor pluginAccessor;

    @Inject
    public BundlesResource(final BundleContext bundleContext, final UriGenerator uriGenerator, final PluginAccessor pluginAccessor) {
        this.bundleContext = bundleContext;
        this.uriGenerator = uriGenerator;
        this.pluginAccessor = pluginAccessor;
    }

    @GET
    @Produces({MediaType.APPLICATION_JSON})
    public Response getBundles(@QueryParam("q") String query) {
        Optional<String> q = query == null || query.equals("") ? Optional.empty() : Optional.of(query);

        return Response.ok(new BundleListBean(uriGenerator, bundleContext, q)).build();
    }

    private Optional<Response> wrongBundleId(@Nonnull Long bundleId) {
        if (bundleId < 0) {
            return Optional.of(Response.status(Response.Status.BAD_REQUEST)
                    .entity("Wrong Bundle ID")
                    .build());
        }
        return Optional.empty();
    }

    private Optional<Response> bundleFound(@Nonnull Long bundleId, Function<Bundle, Response> action) {
        return Optional.ofNullable(bundleContext.getBundle(bundleId)).map(action);
    }

    private Response doWithBundleId(@Nonnull Long bundleId, Function<Bundle, Response> action) {
        return wrongBundleId(bundleId)
                .orElse(bundleFound(bundleId, action)
                        .orElse(Response.status(Response.Status.NOT_FOUND).build()));
    }

    @GET
    @Path("/{bundleId}")
    @Produces({MediaType.APPLICATION_JSON})
    public Response getBundle(@Nonnull @PathParam("bundleId") Long bundleId) {
        return doWithBundleId(bundleId, b -> Response.ok(new BundleBean(uriGenerator, b, Optional.empty())).build());
    }

    @GET
    @Path("/{bundleId}/requiredWires")
    @Produces({MediaType.APPLICATION_JSON})
    public Response getRequiredWires(@Nonnull @PathParam("bundleId") Long bundleId) {
        return doWithBundleId(bundleId, b -> Response.ok(new BundleExplorer(b).getWires(REQUIRED)).build());
    }

    @GET
    @Path("/{bundleId}/providedWires")
    @Produces({MediaType.APPLICATION_JSON})
    public Response getProvidedWires(@Nonnull @PathParam("bundleId") Long bundleId) {
        return doWithBundleId(bundleId, b -> Response.ok(new BundleExplorer(b).getWires(PROVIDED)).build());
    }

    @GET
    @Path("/{bundleId}/servicesInUse")
    @Produces({MediaType.APPLICATION_JSON})
    public Response getServicesInUse(@Nonnull @PathParam("bundleId") Long bundleId) {
        return doWithBundleId(bundleId, b -> Response.ok(new BundleExplorer(b).getServices(REQUIRED)).build());
    }

    @GET
    @Path("/{bundleId}/registeredServices")
    @Produces({MediaType.APPLICATION_JSON})
    public Response getRegisteredServices(@Nonnull @PathParam("bundleId") Long bundleId) {
        return doWithBundleId(bundleId, b -> Response.ok(new BundleExplorer(b).getServices(PROVIDED)).build());
    }

    @GET
    @Path("/{bundleId}/declaredRequirements")
    @Produces({MediaType.APPLICATION_JSON})
    public Response getDeclaredRequirements(@Nonnull @PathParam("bundleId") Long bundleId) {
        return doWithBundleId(bundleId, b -> Response.ok(new BundleExplorer(b).getDeclaredRequirements()).build());
    }

    @GET
    @Path("/{bundleId}/declaredCapabilities")
    @Produces({MediaType.APPLICATION_JSON})
    public Response getDeclaredCapabilities(@Nonnull @PathParam("bundleId") Long bundleId) {
        return doWithBundleId(bundleId, b -> Response.ok(new BundleExplorer(b).getDeclaredCapabilities()).build());
    }

    @GET
    @Path("/{bundleId}/plugin")
    @Produces({MediaType.APPLICATION_JSON})
    public Response getPlugin(@Nonnull @PathParam("bundleId") Long bundleId) {
        return doWithBundleId(bundleId, b -> Response.ok(new PluginBean(OsgiHeaderUtil.getPluginKey(b), pluginAccessor)).build());
    }

    @GET
    @Path("/plugins")
    @Produces({MediaType.APPLICATION_JSON})
    public Response getToPlugins() {
        return Response.ok(Arrays.stream(bundleContext.getBundles())
                .collect(Collectors.toMap(Bundle::getBundleId, b -> new PluginBean(OsgiHeaderUtil.getPluginKey(b), pluginAccessor))))
                .build();
    }
}
