package com.atlassian.plugin.spring.scanner.core;

import com.google.common.annotations.VisibleForTesting;
import org.reflections.Configuration;
import org.reflections.Reflections;
import org.reflections.util.ConfigurationBuilder;
import org.reflections.util.FilterBuilder;
import org.slf4j.Logger;
import org.springframework.stereotype.Component;

import javax.annotation.Nonnull;
import javax.inject.Named;
import java.util.ArrayList;
import java.util.List;

import static java.util.Collections.unmodifiableList;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * Scans the plugin code for well-known injection annotations such as {@link Component}, {@link Named}, etc.
 * <p>
 * This scanner will examine every class file in the build directory for annotations of interest.
 * <p>
 * This class uses a mix of higher level Reflections code to get class information and lower level Javassist byte code
 * helpers to get the rest of the information. We prefer the former, but end up having to use the latter as well.
 */
public class AtlassianSpringByteCodeScanner {

    /**
     * Factory method.
     *
     * @param config the scanner configuration
     * @return a new instance
     * @since 2.1.17
     */
    @Nonnull
    public static AtlassianSpringByteCodeScanner getInstance(final ByteCodeScannerConfiguration config) {
        final AnnotationValidator annotationValidator = new AnnotationValidator();
        final Logger log = config.getLog();
        final SpringIndexWriter springIndexWriter = new SpringIndexWriter(config.getOutputDirectory());
        final ClassScanner reflectionsScanner = new ClassScanner(annotationValidator, new JavassistHelper(),
                log, new ProfileFinder(config.getClassPathUrls(), log), springIndexWriter, config.isVerbose()
        );
        return new AtlassianSpringByteCodeScanner(annotationValidator, config, reflectionsScanner, springIndexWriter);
    }

    private final AnnotationValidator annotationValidator;
    private final ByteCodeScannerConfiguration configuration;
    private final ClassScanner scanner;
    private final SpringIndexWriter springIndexWriter;

    @VisibleForTesting
    AtlassianSpringByteCodeScanner(
            final AnnotationValidator annotationValidator,
            final ByteCodeScannerConfiguration configuration,
            final ClassScanner scanner,
            final SpringIndexWriter springIndexWriter) {
        this.annotationValidator = requireNonNull(annotationValidator);
        this.configuration = requireNonNull(configuration);
        this.scanner = requireNonNull(scanner);
        this.springIndexWriter = requireNonNull(springIndexWriter);
    }

    /**
     * Performs the scanning.
     *
     * @return the results
     * @since 2.1.17 was in the constructor
     */
    public ScanResults scan() {
        final Configuration reflectionsConfig = configureReflections();

        // trigger the Reflections scanner
        new Reflections(reflectionsConfig);

        // write the indexes generated by the scan
        springIndexWriter.writeIndexes();

        return getScanResults();
    }

    private Configuration configureReflections() {
        final ConfigurationBuilder reflectionsConfig = new ConfigurationBuilder();
        reflectionsConfig.setUrls(configuration.getClassPathUrls());
        if (isNotBlank(configuration.getIncludeExclude())) {
            reflectionsConfig.filterInputsBy(FilterBuilder.parse(configuration.getIncludeExclude()));
        }
        reflectionsConfig.setScanners(scanner);
        suppressReflectionsLogging();
        return reflectionsConfig;
    }

    private static void suppressReflectionsLogging() {
        try {
            Reflections.log = null;
        } catch (final Exception e) {
            // Ignore
        }
    }

    private ScanResults getScanResults() {
        final ScanResults results = new ScanResults(scanner);
        if (configuration.shouldValidate()) {
            results.addErrors(annotationValidator.validate());
        }
        return results;
    }

    public static class ScanResults {

        private final int classesEncountered;
        private final int componentClassesEncountered;
        private final List<String> errors = new ArrayList<>();

        public ScanResults(final ClassScanner scanner) {
            this.classesEncountered = scanner.getClassesEncountered();
            this.componentClassesEncountered = scanner.getComponentClassesEncountered();
            this.errors.addAll(scanner.getErrors());
        }

        private void addErrors(final List<String> errors) {
            this.errors.addAll(errors);
        }

        public int getClassesEncountered() {
            return classesEncountered;
        }

        public int getComponentClassesEncountered() {
            return componentClassesEncountered;
        }

        public List<String> getErrors() {
            return unmodifiableList(errors);
        }
    }
}
