package com.atlassian.multitenant.spring;

import com.atlassian.multitenant.MultiTenantComponentMap;
import com.atlassian.multitenant.MultiTenantComponentMapBuilder;
import com.atlassian.multitenant.MultiTenantCreator;
import com.atlassian.multitenant.MultiTenantDestroyer;
import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.multitenant.Tenant;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.util.ClassUtils;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.Set;

/**
 * Spring FactoryBean, that returns a proxy to the component that selects the correct component based on the current
 * tenant.  Components are lazily instantiated, and DisposableBeans destroy() methods are called when a tenant is
 * stopped.
 * <p/>
 * If a listener method is called while no context is set, it will set the context and call the method for each tenant.
 */
public class MultiTenantFactoryBean implements FactoryBean, ApplicationContextAware, BeanNameAware
{
    private static final Logger log = Logger.getLogger(MultiTenantFactoryBean.class);

    private Class[] interfaces;
    private Class implementation;
    private FactoryBean factory;
    private boolean lazyLoad = true;
    private Object proxy;
    private ApplicationContext applicationContext;
    private String name;
    private String targetName;

    public MultiTenantFactoryBean()
    {
        System.out.println("Break me");
    }

    public synchronized Object getObject() throws Exception
    {
        if (proxy == null)
        {
            MultiTenantComponentMapBuilder builder = MultiTenantContext.getFactory().createComponentMapBuilder(new FactoryBeanCreator());
            if (!lazyLoad)
            {
                builder.setLazyLoad(MultiTenantComponentMap.LazyLoadStrategy.EAGER_LOAD);
            }
            MultiTenantComponentMap map = builder.construct();
            if (getImplementingInterfaces() != null && getImplementingInterfaces().length > 0)
            {
                Set<Method> invokeForAllMethods;
                if (isApplicationListener())
                {
                    invokeForAllMethods = Collections.singleton(ApplicationListener.class.getDeclaredMethod(
                            "onApplicationEvent", ApplicationEvent.class));
                }
                else
                {
                    invokeForAllMethods = Collections.emptySet();
                }
                proxy = MultiTenantContext.getFactory().createComponent(map, implementation.getClassLoader(),
                        invokeForAllMethods, getImplementingInterfaces());
            }
            else
            {
                proxy = MultiTenantContext.getFactory().createEnhancedComponent(map, implementation);
            }
        }
        return proxy;
    }

    private boolean isApplicationListener()
    {
        for (Class clazz : getImplementingInterfaces())
        {
            if (clazz.equals(ApplicationListener.class))
            {
                return true;
            }
        }
        return false;
    }

    private Class[] getImplementingInterfaces()
    {
        if (interfaces == null)
        {
            if (implementation == null)
            {
                interfaces = new Class[] { };
            }
            else
            {
                if (FactoryBean.class.isAssignableFrom(implementation))
                {
                    // Try to use the getObjectType of the FactoryBean
                    try
                    {
                        factory = (FactoryBean) implementation.newInstance();
                        implementation = factory.getObjectType();
                        if (implementation == null)
                        {
                            throw new IllegalArgumentException("FactoryBean can only be stateful if it returns an object type");
                        }
                    }
                    catch (InstantiationException e)
                    {
                        throw new IllegalArgumentException("FactoryBean can only be stateful if it can be instantiated with"
                                + "a noarg constructor", e);
                    }
                    catch (IllegalAccessException e)
                    {
                        throw new IllegalArgumentException("FactoryBean can only be stateful if it can be instantiated with"
                                + "a noarg constructor", e);
                    }
                }
                interfaces = implementation.getInterfaces();
            }
        }
        return interfaces;
    }

    public Class getObjectType()
    {
        if (getImplementingInterfaces() == null && getImplementingInterfaces().length == 0)
        {
            return implementation;
        }
        else if (getImplementingInterfaces().length == 1)
        {
            return getImplementingInterfaces()[0];
        }
        return ClassUtils.createCompositeInterface(getImplementingInterfaces(), implementation.getClassLoader());
    }

    public boolean isSingleton()
    {
        return true;
    }

    public void setInterfaces(final Class[] interfaces)
    {
        this.interfaces = interfaces;
    }

    public void setImplementation(final Class implementation)
    {
        this.implementation = implementation;
    }

    public void setApplicationContext(final ApplicationContext applicationContext)
    {
        this.applicationContext = applicationContext;
    }

    public void setBeanName(final String name)
    {
        this.name = name;
    }

    /**
     * Set the name of a prototyped scoped bean to be looked up for each tenant.  If set, the bean with this name will
     * be looked up from the applicationContext, otherwise, the factory will attempt to create a new autowired bean.
     *
     * @param targetName The name of a prototype scoped bean
     */
    public void setTargetName(final String targetName)
    {
        this.targetName = targetName;
    }

    public void setLazyLoad(final boolean lazyLoad)
    {
        this.lazyLoad = lazyLoad;
    }

    private class FactoryBeanCreator
            implements MultiTenantCreator<Object>, MultiTenantDestroyer<Object>
    {
        public Object create(final Tenant tenant)
        {
            boolean contextSet = false;
            // If we're not being lazy loaded, and we don't have a context, set one
            if (!lazyLoad && !MultiTenantContext.getTenantReference().isSet())
            {
                contextSet = true;
                MultiTenantContext.getTenantReference().set(tenant, true);
            }
            try
            {
                if (targetName != null)
                {
                    return applicationContext.getBean(targetName);
                }
                if (factory != null)
                {
                    try
                    {
                        return factory.getObject();
                    }
                    catch (Exception e)
                    {
                        throw new RuntimeException("Exception thrown while trying to instantiate " + name + " from factory", e);
                    }
                }
                return applicationContext.getAutowireCapableBeanFactory().createBean(implementation,
                        AutowireCapableBeanFactory.AUTOWIRE_AUTODETECT, false);
            }
            finally
            {
                if (contextSet)
                {
                    MultiTenantContext.getTenantReference().remove();
                }
            }
        }

        public void destroy(final Tenant tenant, final Object instance)
        {
            if (instance instanceof DisposableBean)
            {
                try
                {
                    ((DisposableBean) instance).destroy();
                }
                catch (Exception e)
                {
                    log.error("Exception thrown while disposing bean " + name + " for tenant " + tenant.getName());
                }
            }
        }
    }


}
