package com.atlassian.multitenant.quartz;

import org.quartz.SchedulerConfigException;
import org.quartz.simpl.SimpleThreadPool;
import org.quartz.spi.ThreadPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Controls the system thread pool.  The system thread pool is used by multi tenant thread pools so they share the
 * same thread pool.
 */
public class SystemThreadPoolController
{
    private static final Logger log = LoggerFactory.getLogger(SystemThreadPoolController.class);
    private static SystemThreadPoolController INSTANCE;

    private final SimpleThreadPool threadPool = new SimpleThreadPool();
    private final SystemThreadPoolWrapper wrapper = new SystemThreadPoolWrapper();

    private SystemThreadPoolController()
    {
        log.debug("New system thread pool controller created.");
    }

    /**
     * Get the instance of this thread pool.  The lifecycle of this pool must be managed manually by the application.
     *
     * @return The instance of this thread pool.
     */
    public static synchronized SystemThreadPoolController getInstance()
    {
        if (INSTANCE == null)
        {
            INSTANCE = new SystemThreadPoolController();
        }
        return INSTANCE;
    }

    /**
     * Initialise the thread pool
     *
     * @param threadCount The number of threads in the thread pool
     * @param threadPriority The priority of the threads
     * @throws SchedulerConfigException If an error occured
     */
    public void initialise(int threadCount, int threadPriority) throws SchedulerConfigException
    {
        log.debug("Starting system thread pool controller");
        threadPool.setThreadCount(threadCount);
        threadPool.setThreadPriority(threadPriority);
        threadPool.setThreadNamePrefix("QuartzWorker");
        threadPool.initialize();
        log.debug("System thread pool controller started");
    }

    /**
     * Shut the thread pool down.
     */
    public void shutdown()
    {
        log.debug("Stopping system thread pool controller");
        threadPool.shutdown(false);
        log.debug("System thread pool controller stopped");
    }

    /**
     * Get the system thread pool.
     *
     * @return The system thread pool
     */
    public ThreadPool getSystemThreadPool()
    {
        return wrapper;
    }

    /**
     * This wraps the system thread pool so that dangerous methods can't be called on it.
     */
    private class SystemThreadPoolWrapper implements ThreadPool
    {
        public boolean runInThread(final Runnable runnable)
        {
            return threadPool.runInThread(runnable);
        }

        public int blockForAvailableThreads()
        {
            return threadPool.blockForAvailableThreads();
        }

        public void setInstanceId(String s)
        {
            // Ignore, we don't care about the instance id of the schedulers
        }

        public void setInstanceName(String s)
        {
            // Ignore, we don't care about the name of the schedulers
        }

        public void initialize() throws SchedulerConfigException
        {
            throw new UnsupportedOperationException("Do not initialise the system thread pool directly, use SystemThreadPoolController.initialise()");
        }

        public void shutdown(final boolean b)
        {
            throw new UnsupportedOperationException("Do not shutdown the system thread pool directly, use SystemThreadPoolController.shutdown()");
        }

        public int getPoolSize()
        {
            return threadPool.getPoolSize();
        }
    }
}
