package com.atlassian.multitenant.quartz;

import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.multitenant.Tenant;
import org.quartz.SchedulerConfigException;
import org.quartz.spi.ThreadPool;

/**
 * Thread pool that ensures that all threads created have the right tenant set, and uses a shared static thread pool
 */
public class MultiTenantThreadPool implements ThreadPool
{
    private final Tenant tenant;
    private final ThreadPool delegate;

    public MultiTenantThreadPool()
    {
        tenant =  MultiTenantContext.getTenantReference().get();
        delegate = SystemThreadPoolController.getInstance().getSystemThreadPool();
    }

    public boolean runInThread(final Runnable runnable)
    {
        return delegate.runInThread(new MultiTenantRunnable(runnable));
    }

    public int blockForAvailableThreads()
    {
        return delegate.blockForAvailableThreads();
    }

    public void initialize() throws SchedulerConfigException
    {
        // Do nothing
    }

    public void shutdown(final boolean waitForJobsToComplete)
    {
        // This is going to be hard to implement if we have to wait for jobs to complete, but for now don't worry about
        // waiting for jobs to complete.
    }

    public int getPoolSize()
    {
        return delegate.getPoolSize();
    }

    public void setInstanceId(String schedInstId)
    {
        // Ignore, we don't care about the instance id of the schedulers
    }

    public void setInstanceName(String schedName)
    {
        // Ignore, we don't care about the name of the schedulers
    }

    private class MultiTenantRunnable implements Runnable
    {
        private final Runnable runnableDelegate;

        private MultiTenantRunnable(final Runnable runnableDelegate)
        {
            this.runnableDelegate = runnableDelegate;
        }

        public void run()
        {
            MultiTenantContext.getTenantReference().set(tenant, false);
            try
            {
                runnableDelegate.run();
            }
            finally
            {
                MultiTenantContext.getTenantReference().remove();
            }
        }
    }
}
