package com.atlassian.multitenant.quartz;

import com.atlassian.multitenant.MultiTenantComponentMap;
import com.atlassian.multitenant.MultiTenantCreator;
import com.atlassian.multitenant.MultiTenantDestroyer;
import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.multitenant.Tenant;
import org.apache.log4j.Logger;
import org.quartz.JobDetail;
import org.quartz.ObjectAlreadyExistsException;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerListener;
import org.quartz.impl.StdSchedulerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.scheduling.quartz.JobDetailAwareTrigger;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.Callable;

/**
 * Factory for instantiating multi tenant scheduler instances.  The reason why we can't simply extend the Spring
 * QuartzSchedulerFactoryBean is because it is inherently singleton, using a single scheduler factory instantiated after
 * properties set, where we need schedule factory per tenant.  This factory returns a singleton proxy to the schedulers,
 * and does not need to be marked as stateful.
 * <p/>
 * This factory assumes that the lifecycle of the scheduler will be managed by the application (ie, the scheduler will
 * not be started automatically).
 */
public class MultiTenantQuartzSchedulerFactoryBean implements FactoryBean, InitializingBean, DisposableBean
{
    private static final Logger log = Logger.getLogger(MultiTenantQuartzSchedulerFactoryBean.class);
    private static final String SCHEDULER_NAME_PREFIX = "multitenant.";

    private Scheduler proxy;
    private MultiTenantComponentMap<Scheduler> map;

    private Properties quartzProperties;
    private List<Trigger> triggers;
    private Map<Object, Object> schedulerContextAsMap;
    private TriggerListener[] globalTriggerListeners;
    private int threadCount = 10;
    private int threadPriority = 4;

    public Object getObject() throws Exception
    {
        return proxy;
    }

    public Class getObjectType()
    {
        return Scheduler.class;
    }

    public boolean isSingleton()
    {
        return true;
    }

    public void afterPropertiesSet() throws Exception
    {
        SystemThreadPoolController.getInstance().initialise(threadCount, threadPriority);
        map = MultiTenantContext.getFactory().createComponentMapBuilder(new QuartzSchedulerCreator())
                .setLazyLoad(MultiTenantComponentMap.LazyLoadStrategy.EAGER_LOAD).construct();
        proxy = MultiTenantContext.getFactory().createComponent(map, Scheduler.class);
    }

    public void destroy() throws Exception
    {
        for (Scheduler scheduler : map.getAll())
        {
            if (scheduler.isStarted())
            {
                scheduler.shutdown();
            }
        }
        SystemThreadPoolController.getInstance().shutdown();
    }

    public void setQuartzProperties(final Properties quartzProperties)
    {
        this.quartzProperties = quartzProperties;
    }

    public void setTriggers(final List<Trigger> triggers)
    {
        this.triggers = triggers;
    }

    public void setSchedulerContextAsMap(final Map<Object, Object> schedulerContextAsMap)
    {
        this.schedulerContextAsMap = schedulerContextAsMap;
    }

    public void setGlobalTriggerListeners(final TriggerListener[] globalTriggerListeners)
    {
        this.globalTriggerListeners = globalTriggerListeners;
    }

    private class QuartzSchedulerCreator
            implements MultiTenantCreator<Scheduler>, MultiTenantDestroyer<Scheduler>
    {
        public Scheduler create(final Tenant tenant)
        {
            try
            {
                final StdSchedulerFactory factory = new StdSchedulerFactory();
                Properties mergedProps = new Properties();
                if (quartzProperties != null)
                {
                    CollectionUtils.mergePropertiesIntoMap(quartzProperties, mergedProps);
                }
                mergedProps.setProperty(StdSchedulerFactory.PROP_SCHED_INSTANCE_NAME, SCHEDULER_NAME_PREFIX + tenant.getName());
                mergedProps.setProperty(StdSchedulerFactory.PROP_THREAD_POOL_CLASS, MultiTenantThreadPool.class.getName());
                factory.initialize(mergedProps);

                // This must run in the correct context, as this call will start the thread pool, which requires
                // the context be set in the running thread.
                // This must override the current context as during start up, there is an initialising context.
                Scheduler scheduler = MultiTenantContext.getManager().callForTenant(tenant, new Callable<Scheduler>()
                {
                    public Scheduler call() throws Exception
                    {
                        return factory.getScheduler();
                    }
                }, true);
                if (schedulerContextAsMap != null)
                {
                    scheduler.getContext().putAll(schedulerContextAsMap);
                }

                if (globalTriggerListeners != null)
                {
                    for (TriggerListener listener : globalTriggerListeners)
                    {
                        scheduler.addGlobalTriggerListener(listener);
                    }
                }

                if (triggers != null)
                {
                    for (Trigger trigger : triggers)
                    {
                        addTriggerToScheduler(scheduler, trigger);
                    }
                }
                scheduler.startDelayed(1000);
                return scheduler;
            }
            catch (SchedulerException se)
            {
                throw new RuntimeException("Error initialising scheduler", se);
            }
            catch (Exception e)
            {
                // thrown by the Callable
                throw new RuntimeException("Unexpected error initialising scheduler", e);
            }
        }

        public void destroy(final Tenant tenant, final Scheduler instance)
        {
            try
            {
                if (instance.isStarted())
                {
                    instance.shutdown();
                }
            }
            catch (SchedulerException se)
            {
                log.warn("Error shutting down scheduler for tenant: " + tenant.getName());
            }
        }
    }

    private void addTriggerToScheduler(Scheduler scheduler, Trigger trigger) throws SchedulerException
    {
        boolean triggerExists = (scheduler.getTrigger(trigger.getName(), trigger.getGroup()) != null);
        if (!triggerExists)
        {
            // Check if the Trigger is aware of an associated JobDetail.
            if (trigger instanceof JobDetailAwareTrigger)
            {
                JobDetail jobDetail = ((JobDetailAwareTrigger) trigger).getJobDetail();
                // Automatically register the JobDetail too.
                addJobToScheduler(scheduler, jobDetail);
            }
            try
            {
                scheduler.scheduleJob(trigger);
            }
            catch (ObjectAlreadyExistsException ex)
            {
                if (log.isDebugEnabled())
                {
                    log.debug("Unexpectedly found existing trigger, assumably due to cluster race condition: " +
                            ex.getMessage() + " - can safely be ignored");
                }
            }
        }
    }

    private void addJobToScheduler(Scheduler scheduler, JobDetail jobDetail) throws SchedulerException
    {
        if (scheduler.getJobDetail(jobDetail.getName(), jobDetail.getGroup()) == null)
        {
            scheduler.addJob(jobDetail, true);
        }
    }

    public void setThreadCount(final int threadCount)
    {
        this.threadCount = threadCount;
    }

    public void setThreadPriority(final int threadPriority)
    {
        this.threadPriority = threadPriority;
    }
}
