package com.atlassian.multitenant.juc;

import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.multitenant.Tenant;
import org.apache.log4j.Logger;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.AbstractExecutorService;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

/**
 * Wrapper for executor services
 */
public class MultiTenantExecutors
{
    private static final Logger log = Logger.getLogger(MultiTenantExecutors.class);

    /**
     * Wrap the supplied ExecutorService in an executorService that will execute their supplied tasks with the same
     * multi tenant context as the thread that submitted the task
     *
     * @param executorService The executor to wrap
     * @return The wrapped executor service
     */
    public static ExecutorService wrap(ExecutorService executorService)
    {
        if (MultiTenantContext.isEnabled())
        {
            return new MultiTenantExecutorServiceDelegate(executorService);
        }
        else
        {
            return executorService;
        }
    }

    /**
     * Wrap the supplied ScheduledExecutorService in an scheduledExecutorService that will execute their supplied tasks
     * with the same multi tenant context as the thread that submitted the task
     *
     * @param scheduledExecutorService The executor to wrap
     * @return The wrapped scheduled executor service
     */
    public static ScheduledExecutorService wrap(ScheduledExecutorService scheduledExecutorService)
    {
        if (MultiTenantContext.isEnabled())
        {
            return new MultiTenantScheduledExecutorServiceDelegate(scheduledExecutorService);
        }
        else
        {
            return scheduledExecutorService;
        }
    }

    private static class MultiTenantScheduledExecutorServiceDelegate extends MultiTenantExecutorServiceDelegate
            implements ScheduledExecutorService
    {
        private final ScheduledExecutorService target;

        private MultiTenantScheduledExecutorServiceDelegate(final ScheduledExecutorService target)
        {
            super(target);
            this.target = target;
        }

        public ScheduledFuture<?> schedule(final Runnable command, final long delay, final TimeUnit unit)
        {
            return target.schedule(new WrappedRunnable(command), delay, unit);
        }

        public <V> ScheduledFuture<V> schedule(final Callable<V> callable, final long delay, final TimeUnit unit)
        {
            return target.schedule(new WrappedCallable<V>(callable), delay, unit);
        }

        public ScheduledFuture<?> scheduleAtFixedRate(final Runnable command, final long initialDelay, final long period, final TimeUnit unit)
        {
            return target.scheduleAtFixedRate(new WrappedRunnable(command), initialDelay, period, unit);
        }

        public ScheduledFuture<?> scheduleWithFixedDelay(final Runnable command, final long initialDelay, final long delay, final TimeUnit unit)
        {
            return target.scheduleWithFixedDelay(new WrappedRunnable(command), initialDelay, delay, unit);
         }
    }

    private static class MultiTenantExecutorServiceDelegate extends AbstractExecutorService
    {
        private final ExecutorService target;

        private MultiTenantExecutorServiceDelegate(final ExecutorService target)
        {
            this.target = target;
        }

        public void shutdown()
        {
            target.shutdown();
        }

        public List<Runnable> shutdownNow()
        {
            List<Runnable> wrappedRunnables = target.shutdownNow();
            List<Runnable> results = new ArrayList<Runnable>(wrappedRunnables.size());
            for (Runnable runnable : wrappedRunnables)
            {
                if (runnable instanceof WrappedRunnable)
                {
                    results.add(((WrappedRunnable) runnable).getTarget());
                }
                else
                {
                    // This will probably happen with scheduled callables, which get wrapped in a runnable
                    results.add(runnable);
                }
            }
            return results;
        }

        public boolean isShutdown()
        {
            return target.isShutdown();
        }

        public boolean isTerminated()
        {
            return target.isTerminated();
        }

        public boolean awaitTermination(final long timeout, final TimeUnit unit) throws InterruptedException
        {
            return target.awaitTermination(timeout, unit);
        }

        public void execute(final Runnable command)
        {
            target.execute(new WrappedRunnable(command));
        }
    }

    private static class WrappedRunnable implements Runnable
    {
        private final Runnable target;
        private final Tenant tenant;

        private WrappedRunnable(final Runnable target)
        {
            this.target = target;
            this.tenant = MultiTenantContext.getTenantReference().get();
        }

        public void run()
        {
            MultiTenantContext.getManager().runForTenant(tenant, target, false);
        }

        public Runnable getTarget()
        {
            return target;
        }
    }

    private static class WrappedCallable<T> implements Callable<T>
    {
        private final Callable<T> target;
        private final Tenant tenant;

        private WrappedCallable(final Callable<T> target)
        {
            this.target = target;
            this.tenant = MultiTenantContext.getTenantReference().get();
        }

        public T call() throws Exception
        {
            return MultiTenantContext.getManager().callForTenant(tenant, target, false);
        }
    }
}
