package com.atlassian.multitenant.hibernate2;

import com.atlassian.multitenant.MultiTenantCreator;
import com.atlassian.multitenant.MultiTenantDestroyer;
import com.atlassian.multitenant.MultiTenantComponentMap;
import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.multitenant.Tenant;
import net.sf.hibernate.cache.Cache;
import net.sf.hibernate.cache.CacheException;
import net.sf.hibernate.cache.CacheProvider;
import net.sf.hibernate.cache.EhCacheProvider;
import net.sf.hibernate.cache.Timestamper;
import org.apache.log4j.Logger;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.Properties;
import java.util.Set;

/**
 * Hibernate EhCache provider for multi tenant instances
 */
public class MultiTenantEhCacheProvider implements CacheProvider
{
    private static final Logger log = Logger.getLogger(MultiTenantEhCacheProvider.class);

    private final MultiTenantComponentMap<CacheProvider> map = MultiTenantContext.getFactory().createComponentMap(
            new CacheProviderCreator());
    private volatile boolean started;
    private volatile Properties properties;
    private static final Set<Method> INVOKE_FOR_ALL_METHODS;

    static
    {
        // When destroy() is called, it must be invoked for all tenants if no tenant context is set.
        Set<Method> methods;
        try
        {
            methods = Collections.singleton(Cache.class.getDeclaredMethod("destroy"));
        }
        catch (NoSuchMethodException nsme)
        {
            // This should never happen
            log.fatal("Could not find destroy method on Hibernate Cache class", nsme);
            methods = Collections.emptySet();
        }
        INVOKE_FOR_ALL_METHODS = methods;
    }

    public Cache buildCache(final String regionName, final Properties properties) throws CacheException
    {
        return MultiTenantContext.getFactory().createComponent(MultiTenantContext.getFactory().createComponentMap(
                new CacheCreator(regionName, properties)), INVOKE_FOR_ALL_METHODS, Cache.class);
    }

    public long nextTimestamp()
    {
        // Whenever hibernate opens a session, this gets called.  There are times when hibernate will open a session
        // during start up because a bean has transaction AOP configured on it, but no actual database access will be
        // done.  If database access is done, an exception will be thrown, so that's ok, from this method, let's
        // allow it to proceed.  What we are doing here is the implementation of the ehcache nextTimestamp() method,
        // which calls static state (so all timestamps are shared across all caches/tenants)
        return Timestamper.next();
    }

    public synchronized void start(final Properties properties) throws CacheException
    {
        this.properties = properties;
        for (CacheProvider cacheProvider : map.getAll())
        {
            cacheProvider.start(properties);
        }
        started = true;
    }

    public synchronized void stop()
    {
        started = false;
        for (CacheProvider cacheProvider : map.getAll())
        {
            cacheProvider.stop();
        }
    }

    private class CacheProviderCreator implements MultiTenantCreator<CacheProvider>,
            MultiTenantDestroyer<CacheProvider>
    {
        public CacheProvider create(final Tenant tenant)
        {
            // Synchronise on the parent class in case we're starting or shutting down in another thread.
            synchronized (MultiTenantEhCacheProvider.this)
            {
                CacheProvider cacheProvider = new EhCacheProvider();
                if (started)
                {
                    try
                    {
                        cacheProvider.start(properties);
                    }
                    catch (CacheException ce)
                    {
                        throw new RuntimeException("Error starting cache", ce);
                    }
                }
                return cacheProvider;
            }
        }

        public void destroy(final Tenant tenant, final CacheProvider instance)
        {
            synchronized (MultiTenantEhCacheProvider.this)
            {
                if (started)
                {
                    instance.stop();
                }
            }
        }
    }

    private class CacheCreator
            implements MultiTenantCreator<Cache>, MultiTenantDestroyer<Cache>
    {
        private final String regionName;
        private final Properties properties;

        private CacheCreator(final String regionName, final Properties properties)
        {
            this.regionName = regionName;
            this.properties = properties;
        }

        public Cache create(final Tenant tenant)
        {
            try
            {
                return map.get().buildCache(regionName, properties);
            }
            catch (CacheException ce)
            {
                throw new RuntimeException("Unable to create cache", ce);
            }
        }

        public void destroy(final Tenant tenant, final Cache instance)
        {
            try
            {
                instance.destroy();
            }
            catch (CacheException ce)
            {
                throw new RuntimeException("Unable to destroy cache", ce);
            }
        }
    }
}
