package com.atlassian.messagequeue;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nullable;
import java.io.Serializable;

import static java.util.Objects.requireNonNull;

/**
 * Message runner key
 * @since 1.0
 */
@PublicApi
public final class MessageRunnerKey implements Serializable, Comparable<MessageRunnerKey> {
    private static final long serialVersionUID = 1L;

    private final String key;

    private MessageRunnerKey(final String key) {
        this.key = requireNonNull(key, "key");
    }

    /**
     * Wraps the provided string as a {@code MessageRunnerKey}.
     * <p>
     * Although it is not necessary for correctness, it will usually make sense to create a single instance of
     * the {@code MessageRunnerKey} and reuse it as a constant, as in:
     * </p>
     * <pre>
     * {@code
     *     private static final MessageRunnerKey <strong>POP3_SERVICE</strong> = MessageRunnerKey.of("com.example.plugin.Pop3Service");
     *
     *     // ...
     *
     *     private void registerJobRunner()
     *     {
     *         schedulerService.registerJobRunner(<strong>POP3_SERVICE</strong>, new Pop3JobRunner());
     *     }
     *
     *     private String scheduleJobWithGeneratedId(String cronExpression)
     *     {
     *         JobConfig jobConfig = JobConfig.forMessageRunnerKey(<strong>POP3_SERVICE</strong>)
     *                  .withSchedule(Schedule.forCronExpression(cronExpression));
     *         return schedulerService.scheduleJobWithGeneratedId(jobConfig);
     *     }
     * }
     * </pre>
     *
     * @param key the job runner key, as a string
     * @return the wrapped job runner key
     */
    public static MessageRunnerKey of(String key) {
        return new MessageRunnerKey(key);
    }

    @Override
    @SuppressWarnings("SimplifiableIfStatement")
    public boolean equals(@Nullable final Object o) {
        if (this == o) {
            return true;
        }
        return o != null
                && o.getClass() == getClass()
                && ((MessageRunnerKey) o).key.equals(key);
    }

    @Override
    public int compareTo(final MessageRunnerKey o) {
        return key.compareTo(o.key);
    }

    @Override
    public int hashCode() {
        return key.hashCode();
    }

    @Override
    public String toString() {
        return key;
    }
}