package com.atlassian.mail.msgraph.service;

import com.atlassian.mail.msgraph.settings.providers.MailConnectionSettingsProvider;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.microsoft.graph.authentication.IAuthenticationProvider;
import com.microsoft.graph.http.GraphServiceException;
import com.microsoft.graph.options.QueryOption;
import com.microsoft.graph.requests.GraphServiceClient;
import com.microsoft.graph.requests.MailFolderCollectionPage;
import com.microsoft.graph.requests.MessageCollectionPage;
import com.microsoft.graph.requests.UserRequestBuilder;
import io.atlassian.fugue.Checked;
import io.atlassian.fugue.Either;
import io.atlassian.fugue.Option;
import io.atlassian.fugue.Unit;
import okhttp3.Request;
import org.joda.time.DateTime;
import org.joda.time.format.ISODateTimeFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.internet.MimeMessage;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static io.atlassian.fugue.Either.left;
import static io.atlassian.fugue.Either.right;
import static io.atlassian.fugue.Option.none;

public class MicrosoftGraphMailClient {
    private static final Logger logger = LoggerFactory.getLogger(MicrosoftGraphMailClient.class);

    public static final Integer MS_LIST_MESSAGES_API_PAGE_SIZE = 10;
    public static final Integer HYBRID_MS_EXCHANGE_SERVER_ERROR = 405;
    public static final String FALLBACK_ERROR_MESSAGE = "Using fallback /me endpoint as /users/ endpoint failed. Please update the correct user principal name.";

    private final GraphServiceClient<Request> graphClient;

    private Option<String> userPrincipalName = none();

    public MicrosoftGraphMailClient(MailConnectionSettingsProvider settingsProvider) {
        this.graphClient = getGraphClient(settingsProvider);
        this.userPrincipalName = Option.option(settingsProvider.getUserName());
    }

    @VisibleForTesting
    protected MicrosoftGraphMailClient(GraphServiceClient<Request> graphClient, String userPrincipalName) {
        this.graphClient = graphClient;
        this.userPrincipalName = Option.option(userPrincipalName);
    }

    @Nonnull
    @VisibleForTesting
    GraphServiceClient<Request> getGraphClient(MailConnectionSettingsProvider settingsProvider) {
        return GraphServiceClient.builder()
                .authenticationProvider(
                        new SimpleAuthProvider(() -> settingsProvider.getAuthenticationToken().getOrThrow(
                                () -> new CouldNotRetrieveAccessTokenException(
                                        String.format("Could not retrieve access token for connection [%s]", settingsProvider.getMailSettings().getId())
                                )
                        ))
                )
                .buildClient();
    }

    /**
     * Execute using /users endpoint and if it fails try using /me as a fallback.
     * /me is unsupported by Client Credentials and if /me works then it means there is a configuration error
     * and should be fixed by the customer to use the right account name.
     */
    @Nonnull
    public Either<Throwable, MessageCollectionPage> getMessages(DateTime pullFromDate, String folderId) {
        Function<UserRequestBuilder, MessageCollectionPage> getMessages = userRequest ->
                userRequest
                        .mailFolders(folderId)
                        .messages()
                        .buildRequest(getGraphClientQueryOptions(pullFromDate))
                        .top(MS_LIST_MESSAGES_API_PAGE_SIZE)
                        .select("id")
                        .get();

        return executeGraphClientRequestWithFallback(getMessages);
    }

    @Nonnull
    public Either<Throwable, MessageCollectionPage> getNextMessagesPage(MessageCollectionPage messagesPage) {
        return executeGraphClientRequest(g -> Objects.requireNonNull(messagesPage.getNextPage())
                .buildRequest()
                .get());
    }

    @Nonnull
    public List<Message> getMessageStubsFromMessagePage(MessageCollectionPage messagesPage) {
        return messagesPage
                .getCurrentPage()
                .stream()
                .flatMap(messageStub -> getMimeMessage(messageStub).toStream())
                .collect(Collectors.toList());
    }

    @Nonnull
    public Either<Throwable, List<Message>> getMessageStubsFromMessagePageNew(MessageCollectionPage messagesPage) {

        List<com.microsoft.graph.models.Message> messageStubs = messagesPage.getCurrentPage();
        if (messageStubs.size() == 0) {
            return right(new ArrayList<>());
        }

        List<Message> messages = new ArrayList<>();
        Either<Throwable, Message> mimeMessage = null;

        for (com.microsoft.graph.models.Message messageStub : messageStubs) {
            mimeMessage = getMimeMessage(messageStub);

            if (mimeMessage.isRight()) {
                messages.add(mimeMessage.right().get());
            }
        }

        if (messages.isEmpty() && mimeMessage.isLeft()) {
            try {
                GraphServiceException exception = (GraphServiceException) mimeMessage.left().get();
                if (exception.getResponseCode() == HYBRID_MS_EXCHANGE_SERVER_ERROR) {
                    return left(mimeMessage.left().get());
                }
            } catch (Exception e) {
                return right(messages);
            }
        }

        return right(messages);
    }

    /**
     * Execute using /users endpoint and if it fails try using /me as a fallback.
     * /me is unsupported by Client Credentials and if /me works then it means there is a configuration error
     * and should be fixed by the customer to use the right account name.
     */
    @Nonnull
    public Either<Throwable, Message> getMimeMessage(com.microsoft.graph.models.Message messageStub) {
        String pathForMimeMessage = userRequestString() + "/messages/" + messageStub.id + "/$value";
        String fallbackPath = "/me/messages/" + messageStub.id + "/$value";

        Properties props = new Properties();
        Session session = Session.getDefaultInstance(props, null);

        Function<Either<Throwable, InputStream>, Either<Throwable, Message>> fetchMimeMessage = eitherResult ->
                eitherResult
                        .flatMap(stream -> {
                            try {
                                Message msg = new MicrosoftMimeMessage(messageStub.id, session, stream);
                                return right(msg);
                            } catch (MessagingException e) {
                                logger.error(String.format("Failed construct MimeMessage from message stream (id: %s)", messageStub.id), e);
                                return left(e);
                            }
                        });

        return fetchMimeMessage.apply(executeGraphClientRequest(g ->
                g.customRequest(pathForMimeMessage, InputStream.class)
                        .buildRequest()
                        .get())
                .left()
                .flatMap(leftError -> {
                    logger.error(FALLBACK_ERROR_MESSAGE, leftError);
                    return executeGraphClientRequest(g ->
                            g.customRequest(fallbackPath, InputStream.class)
                                    .buildRequest()
                                    .get());
                }));
    }

    public Either<Throwable, MailFolderCollectionPage> getFolderIdByName(String name) {

        Function<UserRequestBuilder, MailFolderCollectionPage> getFolderIdByName = userRequest ->
                userRequest
                        .mailFolders()
                        .buildRequest()
                        .filter("displayName eq '" + name + "'")
                        .get();

        return executeGraphClientRequestWithFallback(getFolderIdByName);
    }

    private UserRequestBuilder userRequestBuilder(GraphServiceClient<Request> g) {
        return userPrincipalName
                .map(g::users)
                .getOr(g::me);
    }

    private String userRequestString() {
        return userPrincipalName
                .map(name -> String.format("/users('%s')", name))
                .getOr(() -> "/me");
    }

    /**
     * Execute using /users endpoint and if it fails try using /me as a fallback.
     * /me is unsupported by Client Credentials and if /me works then it means there is a configuration error
     * and should be fixed by the customer to use the right account name.
     */
    public Either<Throwable, Unit> markMessageRead(Message message) {
        if (message instanceof MicrosoftMimeMessage) {
            MicrosoftMimeMessage msMessage = (MicrosoftMimeMessage) message;
            com.microsoft.graph.models.Message markAsRead = new com.microsoft.graph.models.Message();
            markAsRead.isRead = true;

            markAsRead.id = msMessage.microsoftMessageId;
            logger.info(String.format("Marking message with microsoftMessageId (%s) as read", markAsRead.id));

            Function<UserRequestBuilder, com.microsoft.graph.models.Message> markMessageRead = userRequest ->
                    userRequest
                            .messages(msMessage.getMicrosoftMessageId())
                            .buildRequest()
                            .patch(markAsRead);

            return executeGraphClientRequestWithFallback(markMessageRead)
                    .map(m -> Unit.VALUE);
        } else {
            throw new IllegalArgumentException(getClass() + " asked to handle non-Microsoft source message");
        }
    }

    private <T> Either<Throwable, T> executeGraphClientRequest(Function<GraphServiceClient<Request>, T> func) {
        Either<Throwable, T> either = Checked.now(() -> func.apply(graphClient)).toEither().leftMap(l -> (Throwable) l);

        if (either.isLeft()) {
            handleGraphClientResponseException(either.left().get());
        } else if (either.isRight()) {
            Object response = either.right().get();
            if (response != null) handleGraphClientSuccess();
        }

        return either;
    }

    /**
     * Execute using /users endpoint and if it fails try using /me as a fallback.
     * /me is unsupported by Client Credentials and if /me works then it means there is a configuration error
     * and should be fixed by the customer to use the right account name.
     */
    private <T> Either<Throwable, T> executeGraphClientRequestWithFallback(Function<UserRequestBuilder, T> func) {
        Function<GraphServiceClient<Request>, T> usersEndpoint = g -> func.apply(userRequestBuilder(g));
        Function<GraphServiceClient<Request>, T> meEndpoint = g -> func.apply(g.me());

        return executeGraphClientRequest(usersEndpoint)
                .left().flatMap(leftError -> {
                    logger.error(FALLBACK_ERROR_MESSAGE, leftError);
                    return executeGraphClientRequest(meEndpoint);
                });
    }

    private void handleGraphClientResponseException(Throwable th) {
        if (th instanceof GraphServiceException) {
            GraphServiceException ex = (GraphServiceException) th;
            logger.error(String.valueOf(ex));
        } else {
            logger.error(String.valueOf(th));
        }
    }

    private void handleGraphClientSuccess() {
        logger.debug("Graph client request successful");
    }

    private List<QueryOption> getGraphClientQueryOptions(DateTime pullFromDate) {
        String pullFromDateISO = pullFromDate.toString(ISODateTimeFormat.dateTime());

        QueryOption unreadAndReceivedAfterOption = new QueryOption("$filter", "isRead eq false and receivedDateTime ge " + pullFromDateISO);
        QueryOption oldestFirstOption = new QueryOption("$orderBy", "receivedDateTime asc");

        return ImmutableList.of(unreadAndReceivedAfterOption, oldestFirstOption);
    }

    private static class SimpleAuthProvider implements IAuthenticationProvider {

        final Supplier<String> tokenSupplier;

        public SimpleAuthProvider(Supplier<String> tokenSupplier) {
            this.tokenSupplier = tokenSupplier;
        }

        @Nonnull
        @Override
        public CompletableFuture<String> getAuthorizationTokenAsync(@Nonnull URL requestUrl) {
            return CompletableFuture.completedFuture(tokenSupplier.get());
        }

    }

    @VisibleForTesting
    public static class MicrosoftMimeMessage extends MimeMessage {

        private final String microsoftMessageId;

        public MicrosoftMimeMessage(String messageStubId, Session session, InputStream is) throws MessagingException {
            super(session, is);
            this.microsoftMessageId = messageStubId;
        }

        public String getMicrosoftMessageId() {
            return microsoftMessageId;
        }
    }

}
