/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.atlassian.labs.httpservice;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.osgi.util.OsgiHeaderUtil;
import com.atlassian.plugin.servlet.ServletModuleManager;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceFactory;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.http.HttpService;


/**
 *  Basic implementation of OSGi HTTP service 1.1.
 *
 *  TODO:
 *
 *      - fuller suite of testing and compatibility tests
 *
 *      - only exposed params are those defined in the OSGi spec. Jetty is
 *        very tunable via params, some of which it may be useful to expose
 *
 *      - no cacheing is performed on delivered resources. Although not part
 *        of the OSGi spec, it also isn't precluded and would enhance
 *        performance in a high usage environment. Jetty's ResourceHandler
 *        class could be a model for this.
 *
 *      - scanning the Jetty ResourceHandler class it's clear that there are
 *        many other sophisticated areas to do with resource handling such
 *        as checking date and range fields in the http headers. It's not clear
 *        whether any of these play a part in the OSGi service - the spec
 *        just describes "returning the contents of the URL to the client" which
 *        doesn't state what other HTTP handling might be compliant or desirable
 */
public class Activator implements BundleActivator{

    private static final Log log = LogFactory.getLog(Activator.class);
    private ServiceRegistration httpServiceFactory;

    public void start(BundleContext bundleContext) throws Exception
    {
        if (bundleContext.getServiceReference(ServletModuleManager.class.getName()) == null) {
            log.error("The servlet module manager isn't registered as a host component.  The HttpService " +
                      "can not be registered.");
        } else {
            httpServiceFactory = bundleContext.registerService(HttpService.class.getName(), new HttpServiceFactory(), null);
        }
    }

    public void stop(BundleContext bundleContext) throws Exception
    {
        if (httpServiceFactory != null) {
            httpServiceFactory.unregister();
        }
    }

    private static <T> T findService(BundleContext bundleContext, Class<T> type)
    {
        ServiceReference ref = bundleContext.getServiceReference(type.getName());
        if (ref == null)
        {
            throw new IllegalStateException("Unable to locate service of type " + type + ".");
        }
        else
        {
            return (T) bundleContext.getService(ref);
        }
    }

    // Inner class to provide basic service factory functionality
    public class HttpServiceFactory implements ServiceFactory
    {

        public Object getService( Bundle bundle, ServiceRegistration registration )
        {
            // todo: handle reloading host components
            BundleContext bundleContext = bundle.getBundleContext();
            PluginAccessor pluginAccessor = findService(bundleContext, PluginAccessor.class);
            ServletModuleManager servletModuleManager = findService(bundleContext, ServletModuleManager.class);

            String key = OsgiHeaderUtil.getPluginKey(bundle);
            Object srv = new HttpServiceImpl(pluginAccessor.getPlugin(key), bundle, servletModuleManager);
            log.debug( "** http service get:" + bundle + ", service: " + srv );
            return srv;
        }

        public void ungetService( Bundle bundle, ServiceRegistration registration, Object service )
        {
            log.debug( "** http service unget:" + bundle + ", service: " + service );
            ( ( HttpServiceImpl ) service ).unregisterAll();
        }
    }
}
