package com.atlassian.json.marshal.wrapped;

/**
 * Jsonable wrapper for string values
 * @since v0.9
 */
public class JsonableString extends WrappingJsonable<String>
{
    public JsonableString(String value)
    {
        super(value);
    }

    @Override
    protected String convertValueToString(String value)
    {
        return jsonEscape(value);
    }

    // protected for testing
    static String jsonEscape(String input)
    {
        if (input == null) return "\"null\""; // be transparent with the way we behaved before

        StringBuilder sb = new StringBuilder((int)(input.length() * 1.1));
        sb.append("\"");

        for (char c : input.toCharArray())
        {
            switch (c)
            {
                // For these special-case characters, only " and \ strictly need encoding, but other characters are encoded
                // to make debugging easier. The RFC is at http://www.ietf.org/rfc/rfc4627.txt also see
                // http://unicode-table.com/en/sections/ for information about the different unicode sections in a more
                // palatable format.
                case '"' :
                case '/' :
                case '\\' :
                    sb.append('\\');
                    sb.append(c);
                    break;
                case '\b' : sb.append("\\b"); break;
                case '\f' : sb.append("\\f"); break;
                case '\n' : sb.append("\\n"); break;
                case '\r' : sb.append("\\r"); break;
                case '\t' : sb.append("\\t"); break;
                default :
                    // ASCII control characters (0x0000 .. 0x001F) must be escaped for valid json
                    if (c >= '\u0000' && c <= '\u001F') appendHex(c, sb);
                    // Additional Unicode control and formatting characters - some libraries barf on these. We start one
                    // character earlier at delete for the same reason.
                    else if (c >= '\u007F' && c <= '\u009F') appendHex(c, sb);
                    else sb.append(c);
            }
        }

        sb.append("\"");
        return sb.toString();
    }

    private static void appendHex(char c, StringBuilder sb)
    {
        sb.append("\\u");
        String hex = Integer.toHexString((int) c).toUpperCase();
        switch (hex.length())
        {
            case 1: sb.append ("000").append(hex); return;
            case 2: sb.append ("00").append(hex); return;
            case 3: sb.append ("0").append(hex); return;
            case 4: sb.append(hex); return;
        }
        throw new IllegalStateException("Could not calculate unicode for character: " + hex);
    }
}
