(function (Backbone) {
    var PropertiesView = JIRA.WorkflowDesigner.PropertiesPanel.PropertiesView,
        WorkflowGlobalTransitionsAJAXManager = JIRA.WorkflowDesigner.IO.AJAX.WorkflowGlobalTransitionsAJAXManager;

    AJS.namespace("JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView");

    Backbone.define("JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView", PropertiesView.extend(
    /** @lends JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView# */
    {
        events: {
            "change #create-global-transition-sidebar": "_synchroniseGlobalTransition",
            "click .delete": "_destroy",
            "click .edit": "_edit"
        },

        template: JIRA.WorkflowDesigner.Templates.PropertiesPanel.status,

        ui: {
            createGlobalTransition: "#create-global-transition-sidebar"
        },

        /**
         * Initialize the view.
         *
         * @classdesc The content of the properties panel when a status is selected.
         * @constructs
         * @extends JIRA.WorkflowDesigner.PropertiesPanel.PropertiesView
         * @param {object} options
         * @param {Backbone.View} options.selectedView The workflow designer's currently selected view.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The workflow model.
         */
        initialize: function () {
            _.bindAll(this, "_onCreateSuccess", "_onError", "_triggerTransactionEnd");

            PropertiesView.prototype.initialize.apply(this, arguments);
        },

        /**
         * Create a global transition to the status.
         *
         * Triggers a `transaction:end` event on completion.
         *
         * @private
         */
        _createGlobalTransition: function () {
            var options,
                request,
                statusModel = this._selectedView.model;

            options = {
                description: "",
                name: statusModel.get("name"),
                screenId: 0,
                statusId: statusModel.get("statusId"),
                workflowName: this._workflowModel.get("name")
            };

            request = WorkflowGlobalTransitionsAJAXManager.createGlobalTransition(options);
            request.always(this._triggerTransactionEnd).done(this._onCreateSuccess).fail(this._onError);
        },

        /**
         * Delete the global transition associated with the status.
         *
         * Triggers a `transaction:end` event on completion.
         *
         * @private
         */
        _deleteGlobalTransition: function () {
            var globalTransition = this._workflowModel.getGlobalTransitionForStatus(this._selectedView.model),
                instance = this,
                options,
                request;

            function removeTransition() {
                instance._workflowModel.removeTransition(globalTransition);
            }

            options = {
                transitionId: globalTransition.get("actionId"),
                workflowName: this._workflowModel.get("name")
            };

            request = WorkflowGlobalTransitionsAJAXManager.deleteGlobalTransition(options);
            request.always(this._triggerTransactionEnd).done(removeTransition).fail(this._onError);
        },

        /**
         * @private
         * @returns {boolean} Whether the selected status is the target of a global transition.
         */
        _hasGlobalTransition: function () {
            var statusModel = this._selectedView.model;
            return this._workflowModel.statusHasGlobalTransition(statusModel);
        },

        /**
         * Reset the `WorkflowModel` and re-select the status.
         *
         * Called when a global transition is successfully created.
         *
         * @param {object} layoutData The new workflow layout.
         * @private
         */
        _onCreateSuccess: function (layoutData) {
            var statusId = this._selectedView.model.id;

            this._workflowModel.reset(layoutData);
            this._canvasModel.set("selectedModel", this._workflowModel.getStatus(statusId));
        },

        /**
         * Reset the view and show an error message.
         *
         * Called when creating/destroying a global transition fails.
         *
         * @param {string} errorMessage The error message to show.
         * @private
         */
        _onError: function (errorMessage) {
            JIRA.WorkflowDesigner.Messages.showErrorMessage(errorMessage);
            this.ui.createGlobalTransition.attr("checked", this._hasGlobalTransition());
        },

        /**
         * @return {object} The data to pass to the view's template.
         */
        serializeData: function () {
            var workflowName = JIRA.WorkflowDesigner.BrowserUtilities.escapeQueryParameter(this._workflowModel.get("name"));

            return _.extend(PropertiesView.prototype.serializeData.apply(this), {
                hasGlobalTransition: this._hasGlobalTransition(),
                isDraft: this._workflowModel.get("draft"),
                workflowName: workflowName
            });
        },

        /**
         * Create or destroy a global transition to match the state of the checkbox.
         *
         * @private
         */
        _synchroniseGlobalTransition: function () {
            var isChecked;

            isChecked = this.ui.createGlobalTransition.is(":checked");
            isChecked ? this._createGlobalTransition() : this._deleteGlobalTransition();
            this.trigger("transaction:start");
        },

        /**
         * Trigger a `transaction:end` event.
         *
         * @private
         */
        _triggerTransactionEnd: function () {
            this.trigger("transaction:end");
        }
    },

    /** @lends JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView */
    {
        /**
         * @param {Backbone.View} view View to check.
         * @return {boolean} Whether `view` is a {@link JIRA.WorkflowDesigner.StatusView}.
         */
        canHandleView: function (view) {
            return view instanceof JIRA.WorkflowDesigner.StatusView;
        }
    }
    ));
}(JIRA.WorkflowDesigner.Backbone));