AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:test-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer");

module("JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView", {
    /**
     * Check the global transition checkbox in a `StatusPropertiesView`.
     *
     * @param {JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView} view A `StatusPropertiesView`.
     */
    checkGlobalTransitionCheckbox: function (view) {
        view.ui.createGlobalTransition.attr("checked", "");
        view.ui.createGlobalTransition.change();
    },

    /**
     * @param {object} [options] Options to pass to the view's constructor.
     * @returns {JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView} A `StatusPropertiesView`.
     */
    createView: function (options) {
        var statusModel, statusView, workflowModel;

        statusModel = new JIRA.WorkflowDesigner.StatusModel({
            id: "status-id",
            name: "Status Name",
            statusId: "status-id",
            stepId: 1
        });

        statusView = new JIRA.WorkflowDesigner.StatusView({
            canvas: JIRA.WorkflowDesigner.TestUtilities.testDraw2DCanvas(),
            model: statusModel
        });

        workflowModel = new JIRA.WorkflowDesigner.WorkflowModel({
            name: "Workflow Name"
        });

        options = _.defaults({}, options, {
            canvasModel: new JIRA.WorkflowDesigner.CanvasModel({}, {
                workflowModel: workflowModel
            }),
            selectedView: statusView,
            workflowModel: workflowModel
        });

        return new JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView(options).render();
    },

    setup: function () {
        var sandbox = this.sandbox = sinon.sandbox.create();

        this.createGlobalTransitionDeferred = jQuery.Deferred();
        this.createGlobalTransitionStub = sandbox.stub(JIRA.WorkflowDesigner.IO.AJAX.WorkflowGlobalTransitionsAJAXManager, "createGlobalTransition").returns(this.createGlobalTransitionDeferred);
        this.deleteGlobalTransitionDeferred = jQuery.Deferred();
        this.deleteGlobalTransitionStub = this.sandbox.stub(JIRA.WorkflowDesigner.IO.AJAX.WorkflowGlobalTransitionsAJAXManager, "deleteGlobalTransition").returns(this.deleteGlobalTransitionDeferred);
        this.globalTransition = new JIRA.WorkflowDesigner.TransitionModel({actionId: "action-id"});
        this.showErrorMessageSpy = sandbox.stub(JIRA.WorkflowDesigner.Messages, "showErrorMessage");
        this.statusHasGlobalTransitionStub = sandbox.stub(JIRA.WorkflowDesigner.WorkflowModel.prototype, "statusHasGlobalTransition").returns(false);

        this.sandbox.stub(JIRA.WorkflowDesigner.WorkflowModel.prototype, "getGlobalTransitionForStatus").returns(this.globalTransition);
    },

    /**
     * @param {boolean} [isDraft=false] Whether the workflow is a draft.
     * @returns {string} The URL of the status properties page.
     */
    statusPropertiesUrl: function (isDraft) {
        return [
            AJS.contextPath(),
            "/secure/admin/workflows/ViewWorkflowStepMetaAttributes.jspa?workflowMode=",
            isDraft ? "draft" : "live",
            "&workflowName=Workflow+Name&workflowStep=1"
        ].join("");
    },

    teardown: function () {
        this.sandbox.restore();
    },

    /**
     * Uncheck the global transition checkbox in a `StatusPropertiesView`.
     *
     * @param {JIRA.WorkflowDesigner.PropertiesPanel.StatusPropertiesView} view A `StatusPropertiesView`.
     */
    uncheckGlobalTransitionCheckbox: function (view) {
        view.ui.createGlobalTransition.removeAttr("checked");
        view.ui.createGlobalTransition.change();
    }
});

test("Checking the global transition checkbox creates a global transition", function () {
    var canvasModelSetSpy,
        expectedArguments,
        layoutData = {},
        statusModel = new JIRA.WorkflowDesigner.StatusModel(),
        transactionEndSpy = sinon.spy(),
        transactionStartSpy = sinon.spy(),
        view = this.createView(),
        workflowModelGetStatusSpy = this.sandbox.stub(JIRA.WorkflowDesigner.WorkflowModel.prototype, "getStatus").returns(statusModel),
        workflowModelResetSpy = this.sandbox.spy(JIRA.WorkflowDesigner.WorkflowModel.prototype, "reset");

    canvasModelSetSpy = this.sandbox.spy(JIRA.WorkflowDesigner.CanvasModel.prototype, "set");
    expectedArguments = [{
        description: "",
        name: "Status Name",
        screenId: 0,
        statusId: "status-id",
        workflowName: "Workflow Name"
    }];

    view.on({
        "transaction:end": transactionEndSpy,
        "transaction:start": transactionStartSpy
    });

    this.checkGlobalTransitionCheckbox(view);
    equal(this.createGlobalTransitionStub.callCount, 1, "WorkflowGlobalTransitionsAJAXManager.createGlobalTransition was called");
    deepEqual(this.createGlobalTransitionStub.args[0], expectedArguments, "It was passed the correct arguments");
    equal(transactionStartSpy.callCount, 1, "A transaction:start event was triggered");

    this.createGlobalTransitionDeferred.resolve(layoutData);

    ok(canvasModelSetSpy.thisValues[0].get("selectedModel") === statusModel, "The status was re-selected");
    equal(transactionEndSpy.callCount, 1, "A transaction:end event was triggered");
    ok(workflowModelGetStatusSpy.calledWithExactly("status-id"), "The new StatusModel was requested from the WorkflowModel");
    equal(workflowModelResetSpy.callCount, 1, "WorkflowModel#reset was called");
    ok(workflowModelResetSpy.args[0][0] === layoutData, "It was passed the correct layout data");
});

test("Clicking the delete button calls StatusView#destroy", function () {
    var destroyStub = this.sandbox.stub(JIRA.WorkflowDesigner.StatusView.prototype, "destroy"),
        view;

    view = this.createView();
    view.$(".delete").click();
    equal(destroyStub.callCount, 1, "StatusView#destroy was called");
});

test("Contains a link to draft workflow status properties", function () {
    var link, view;

    view = this.createView({
        workflowModel: new JIRA.WorkflowDesigner.WorkflowModel({
            draft: true,
            name: "Workflow Name"
        })
    });

    link = view.$(".status-properties");
    equal(link.attr("href"), this.statusPropertiesUrl(true), "The properties link is correct");
});

test("Contains a link to live workflow status properties", function () {
    var link,
        view = this.createView();

    link = view.$(".status-properties");
    equal(link.attr("href"), this.statusPropertiesUrl(), "The properties link is correct");
});

test("Failure to create a global transition is handled", function () {
    var errorMessage = "No global transition for you!",
        view = this.createView();

    this.checkGlobalTransitionCheckbox(view);
    this.createGlobalTransitionDeferred.reject(errorMessage);

    equal(this.showErrorMessageSpy.callCount, 1, "An error message was shown");
    equal(this.showErrorMessageSpy.args[0][0], errorMessage, "Its text is correct");
    ok(!view.ui.createGlobalTransition.is(":checked"), "The global transition checkbox isn't checked");
});

test("Failure to destroy a global transition is handled", function () {
    var errorMessage = "This global transition stays",
        view;

    this.statusHasGlobalTransitionStub.returns(true);
    view = this.createView();
    this.uncheckGlobalTransitionCheckbox(view);
    this.deleteGlobalTransitionDeferred.reject(errorMessage);

    equal(this.showErrorMessageSpy.callCount, 1, "An error message was shown");
    equal(this.showErrorMessageSpy.args[0][0], errorMessage, "Its text is correct");
    ok(view.ui.createGlobalTransition.is(":checked"), "The global transition checkbox is checked");
});

test("The global transition checkbox is checked if the status has a global transition", function () {
    var view;

    this.statusHasGlobalTransitionStub.returns(true);
    view = this.createView();

    ok(view.ui.createGlobalTransition.is(":checked"), "The global transition checkbox is checked");
});

test("The global transition checkbox isn't checked if the status doesn't have a global transition", function () {
    // statusHasGlobalTransitionStub returns false.
    ok(!this.createView().ui.createGlobalTransition.is(":checked"),
            "The global transition checkbox isn't checked");
});

test("Unchecking the global transition checkbox destroys the status's global transition", function () {
    var expectedArguments,
        removeTransitionSpy = this.sandbox.spy(JIRA.WorkflowDesigner.WorkflowModel.prototype, "removeTransition"),
        transactionEndSpy = sinon.spy(),
        transactionStartSpy = sinon.spy(),
        view;

    expectedArguments = [{
        transitionId: "action-id",
        workflowName: "Workflow Name"
    }];

    this.statusHasGlobalTransitionStub.returns(true);
    view = this.createView();
    view.on({
        "transaction:start": transactionStartSpy,
        "transaction:end": transactionEndSpy
    });

    this.uncheckGlobalTransitionCheckbox(view);
    equal(this.deleteGlobalTransitionStub.callCount, 1, "WorkflowGlobalTransitionsAJAXManager.deleteGlobalTransition was called");
    deepEqual(this.deleteGlobalTransitionStub.args[0], expectedArguments, "It was passed the correct arguments");
    equal(transactionStartSpy.callCount, 1, "A transaction:start event was triggered");

    this.deleteGlobalTransitionDeferred.resolve();
    equal(removeTransitionSpy.callCount, 1, "WorkflowModel#removeTransition was called");
    ok(removeTransitionSpy.args[0][0] === this.globalTransition, "It was passed the correct global transition");
    equal(transactionEndSpy.callCount, 1, "A transaction:end event was triggered");
});
