AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:test-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer");

module("JIRA.WorkflowDesigner.NewTransitionFormView", {
    /**
     * Create and render a `NewTransitionFormView`.
     *
     * The view and its dependencies are seeded with test data.
     *
     * @returns {JIRA.WorkflowDesigner.NewTransitionFormView} The view.
     */
    createView: function () {
        var view;

        view = new JIRA.WorkflowDesigner.NewTransitionFormView({
            model: new JIRA.WorkflowDesigner.TransitionModel(),
            screens: [{text: "Screen", value: 1}],
            sourceStatuses: [{text: "Open", value: 2}],
            targetStatuses: [{text: "Closed", value: 3}],
            workflowModel: new JIRA.WorkflowDesigner.WorkflowModel({
                name: "Workflow"
            })
        });

        this.sandbox.spy(view.model, "set");
        view.options.workflowModel.get("statuses").add([{stepId: 2}, {stepId: 3}]);
        view.render();
        view.onShow();

        return view;
    },

    setup: function () {
        var sandbox = this.sandbox = sinon.sandbox.create();

        this.addTransitionDeferred = jQuery.Deferred();
        this.addTransitionStub = sandbox.stub(JIRA.WorkflowDesigner.IO.AJAX.WorkflowTransitionsAJAXManager, "addTransition").returns(this.addTransitionDeferred);
    },

    teardown: function () {
        this.sandbox.restore();
    }
});

test("Clears the transition's angles on status change", function () {
    var expectedArributes, view;

    expectedArributes = {
        sourceAngle: null,
        targetAngle: null
    };

    view = this.createView();
    view.ui.sourceStepId.trigger("selected");
    ok(view.model.set.getCall(0).calledWithExactly(expectedArributes),
            "The transition's angles were cleared after changing the source status");

    view.ui.targetStepId.trigger("selected");
    ok(view.model.set.getCall(0).calledWithExactly(expectedArributes),
            "The transition's angles were cleared after changing the target status");
});

test("submit() rejects the returned deferred on failure", function () {
    var submitPromise;

    submitPromise = this.createView().submit();
    this.addTransitionDeferred.reject("Error");

    equal(submitPromise.state(), "rejected", "The returned promise was rejected");
    submitPromise.fail(function (errorMessage) {
        equal(errorMessage, "Error", "It was rejected with the correct error message");
    });
});

test("submit() successfully creates a new transition", function () {
    var expectedAddTransitionOptions, expectedAttributes, submitPromise, view;

    view = this.createView();
    expectedAddTransitionOptions = {
        description: "Close the issue.",
        name: "Close",
        screenId: 1,
        sourceStepId: 2,
        targetStepId: 3,
        workflowName: "Workflow"
    };

    expectedAttributes = {
        description: "Close the issue.",
        name: "Close",
        screenId: 1,
        screenName: "Screen",
        source: view.options.workflowModel.get("statuses").at(0),
        target: view.options.workflowModel.get("statuses").at(1)
    };

    JIRA.WorkflowDesigner.TestUtilities.fill(view.el, {
        "#transition-description": "Close the issue.",
        "#transition-name": "Close",
        "#transition-screen-id": 1,
        "#transition-source-step-id": 2,
        "#transition-target-step-id": 3
    });

    submitPromise = view.submit();

    equal(submitPromise.state(), "pending", "The returned promise is pending");
    ok(this.addTransitionStub.calledWithExactly(expectedAddTransitionOptions),
            "WorkflowTransitionsAJAXManager.addTransition was called with the correct arguments");

    this.addTransitionDeferred.resolve();
    equal(submitPromise.state(), "resolved", "The returned promise was resolved");
    ok(view.model.set.calledWithExactly(expectedAttributes),
            "The correct attributes were set on the TransitionModel");
});