(function (Backbone) {
    /**
     * Make a dialog's submit button use ADG's "primary" style.
     *
     * Executed in the context of a `JIRA.FormDialog`.
     */
    function makeSubmitButtonPrimary() {
        this.get$popup().find(":submit").removeClass("button");
    }

    /**
     * @class JIRA.WorkflowDesigner.Dialogs.FormDialogView
     * @classdesc
     * A Backbone view wrapper around `JIRA.FormDialog`.
     *
     * Subclasses must implement `_getContent()` and `_onSubmit()`.
     * @extends Backbone.Marionette.ItemView
     */
    Backbone.define("JIRA.WorkflowDesigner.Dialogs.FormDialogView", Backbone.Marionette.Layout.extend(
    /** @lends JIRA.WorkflowDesigner.Dialogs.FormDialogView# */
    {
        /**
         * Disable all buttons/inputs in the dialog.
         */
        disable: function () {
            this.$(":input").attr("disabled", "disabled");
        },

        /**
         * Enable all buttons/inputs in the dialog.
         */
        enable: function () {
            this.$(":input").removeAttr("disabled");
        },

        /**
         * To be implemented by subclasses.
         *
         * @protected
         * @returns {string} The window title to use when the dialog is visible.
         */
        _getWindowTitle: jQuery.noop,

        /**
         * @method
         */
        hide: function () {
            this.dialog.hide(true);
        },

        /**
         * Called when the dialog's content is refreshed.
         *
         * To be implemented by subclasses.
         * @method
         */
        _onContentRefresh: jQuery.noop,

        /**
         * Called on error.
         *
         * @param {string} message The error message to show in the dialog.
         * @private
         */
        _onError: function (message) {
            this.enable();
            this.showErrorMessage(message);
        },

        /**
         * Called when the dialog is hidden.
         *
         * @param {jQuery.Event} e The hide event.
         * @param {jQuery} element The element that contains the dialog.
         * @param {string} reason The reason why the dialog was hidden.
         * @private
         */
        _onHide: function (e, element, reason) {
            var wasCancelled;

            if (!this._onCancel) {
                return;
            }

            wasCancelled = _.contains([
                JIRA.Dialog.HIDE_REASON.cancel,
                JIRA.Dialog.HIDE_REASON.escape
            ], reason);

            wasCancelled && this._onCancel();
        },

        /**
         * Called when the dialog's form is submitted.
         *
         * To be implemented by subclasses.
         */
        _onSubmit: function () {
            throw new Error("FormDialogView subclasses must implement _onSubmit().");
        },

        /**
         * Remove all error messages from the dialog.
         */
        removeErrorMessages: function () {
            this.$(".error").remove();
        },

        /**
         * Reposition the dialog such that it is in the centre of the screen.
         */
        reposition: function () {
            this.dialog._positionInCenter();
        },

        /**
         * @returns {JIRA.WorkflowDesigner.FormDialogView} `this`
         */
        show: function () {
            var instance = this;

            function onContentRefresh() {
                makeSubmitButtonPrimary.call(this);
                instance.setElement(this.get$popup());
                instance.bindUIElements();
                instance._onContentRefresh();
            }

            this.dialog = new JIRA.FormDialog({
                content: this._getContent(),
                id: this.id,
                onContentRefresh: onContentRefresh,
                stacked: true,
                submitHandler: _.bind(this._onSubmit, this),
                windowTitle: this._getWindowTitle()
            });

            this.dialog.show();
            jQuery(this.dialog).one("Dialog.hide", _.bind(this._onHide, this));
            return this;
        },

        /**
         * Show an error message at the top of the dialog.
         *
         * Any existing error message is removed, the dialog is repositioned,
         * and the vertical scroll is reset. Does nothing if `message` is empty.
         *
         * @param {string} message The error message to show.
         */
        showErrorMessage: function (message) {
            var error = aui.message.error({content: message}),
                formBody = this.$(".form-body");

            if (message) {
                formBody.find(".aui-message.error").remove();
                formBody.prepend(error).scrollTop(0);
                this.reposition();
            }
        }
    }));
}(JIRA.WorkflowDesigner.Backbone));