(function (Backbone) {
    Backbone.define("JIRA.WorkflowDesigner.CanvasView", Backbone.View.extend(
    /** @lends JIRA.WorkflowDesigner.CanvasView# */
    {
        /**
         * CSS class name used for the view's element
         *
         * @type {string}
         * @default
         */
        className: "canvas",

        /**
         * Initialise the view.
         *
         * @classdesc The workflow designer's canvas on which statuses and transitions are drawn.
         * @constructs
         * @extends Backbone.View
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.CanvasModel} options.canvasModel The application's `CanvasModel`.
         * @param {boolean} [options.immutable=false] If true, prevents user modifications to the workflow.
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's `WorkflowModel`.
         */
        initialize: function (options) {
            _.bindAll(this, "_onKeyDown");

            options = _.defaults({}, options, {
                immutable: false
            });

            this._canvasModel = options.canvasModel;
            this.immutable = options.immutable;
            this.statusViews = new JIRA.WorkflowDesigner.Collection();
            this.transitionViews = new JIRA.WorkflowDesigner.Collection();
            this._workflowModel = options.workflowModel;
            this._zoomHandler = new JIRA.WorkflowDesigner.ZoomHandler(this.el);

            this.immutable || jQuery(document).on("keydown", this._onKeyDown);
            this.listenTo(this._canvasModel, "change:selectedModel", this._onSelectedModelChange);
            this.listenTo(this.statusViews, {
                deselected: this._clearSelectedView,
                "port:drag:end": this._onPortDragEnd,
                "port:drag:start": _.bind(this._setPortDragged, this, true),
                selected: this._setSelectedView
            });

            this.listenTo(this.transitionViews, {
                deselected: this._onTransitionDeselected,
                reconnect: this._onReconnect,
                selected: this._onTransitionSelected
            });

            this.listenTo(this._workflowModel, "reset:after", this._positionNewStatuses);
            this.listenTo(this._workflowModel.get("statuses"), {
                add: this.addStatus,
                remove: this.removeStatus,
                reset: this.resetStatuses
            });

            this.listenTo(this._workflowModel.get("transitions"), {
                add: this.addTransition,
                "change:source change:target": this._updateTransitionSourceAndTarget,
                remove: this.removeTransition,
                reset: this.resetTransitions
            });

            this.listenTo(this._zoomHandler, "zoom", this._onZoom);
        },

        /**
         * Add a status to the canvas.
         *
         * @param {JIRA.WorkflowDesigner.StatusModel} statusModel The status to add.
         * @return {JIRA.WorkflowDesigner.StatusView} The view that was added.
         */
        addStatus: function (statusModel) {
            var isInitial = statusModel.get("initial"),
                isSelected = this._canvasModel.get("selectedModel") === statusModel,
                statusView,
                viewClass;

            if (isInitial) {
                viewClass = JIRA.WorkflowDesigner.InitialStatusView;
            } else {
                viewClass = JIRA.WorkflowDesigner.StatusView;
            }

            statusView = new viewClass({
                canvas: this.canvas,
                getAllTargetPorts: _.bind(this._getAllTargetPorts, this),
                immutable: this.immutable,
                isPortDragged: _.bind(this._isPortDragged, this),
                model: statusModel,
                workflowModel: this._workflowModel
            }).render();

            this.statusViews.add(statusView);
            isSelected && statusView.select();
            return statusView;
        },

        /**
         * Add a transition to the canvas, normal or global.
         *
         * Does nothing if the transition is already present.
         *
         * @param {JIRA.WorkflowDesigner.TransitionModel} transitionModel The transition to add.
         * @return {JIRA.WorkflowDesigner.TransitionView} The view that was added.
         */
        addTransition: function (transitionModel) {
            var isGlobalTransition = transitionModel.get("globalTransition"),
                isSelected = this._canvasModel.get("selectedModel") === transitionModel,
                transitionView = this._getTransitionViewWithModel(transitionModel),
                viewClass;

            if (transitionView) {
                return transitionView;
            }

            if (isGlobalTransition) {
                viewClass = JIRA.WorkflowDesigner.GlobalTransitionView;
            } else {
                viewClass = JIRA.WorkflowDesigner.TransitionView;
            }

            transitionView = new viewClass({
                canvas: this.canvas,
                canvasModel: this._canvasModel,
                immutable: this.immutable,
                model: transitionModel,
                sourceView: this._getStatusViewWithModel(transitionModel.get("source")),
                targetView: this._getStatusViewWithModel(transitionModel.get("target")),
                workflowModel: this._workflowModel
            });

            transitionView.requestResponse && transitionView.requestResponse.setHandler("isSelected", this._transitionIsSelected, this);
            transitionView.render();

            this.transitionViews.add(transitionView);
            isSelected && transitionView.select();
            return transitionView;
        },

        /**
         * Adjust the canvas view box to contain all items.
         *
         * Does nothing if there are no items on the canvas.
         */
        autoFit: function () {
            var boundingBox,
                boundingBoxCenter,
                fits,
                padding = 10,
                viewBox;

            // Ensure that the SVG element's size is correct; otherwise auto
            // resizing will kick in and ruin out beautiful view box.
            this.canvas.fitToContainer();
            boundingBox = this.getCanvasBoundingBox();

            if (boundingBox) {
                boundingBox.scale(padding * 2, padding * 2);
                viewBox = new draw2d.geo.Rectangle(0, 0, this.$el.width(), this.$el.height());

                // Does the bounding box fit in the ideal view box?
                fits = boundingBox.getHeight() <= viewBox.getHeight() &&
                    boundingBox.getWidth() <= viewBox.getWidth();

                if (!fits) {
                    this.canvas.setViewBox(boundingBox);
                    viewBox = this.canvas.getViewBox();
                }

                // Center the diagram in the view box.
                boundingBoxCenter = boundingBox.getCenter();
                this.canvas.setViewBox(new draw2d.geo.Rectangle(
                    boundingBoxCenter.getX() - viewBox.getWidth()/2,
                    boundingBoxCenter.getY() - viewBox.getHeight()/2,
                    viewBox.getWidth(),
                    viewBox.getHeight()
                ));

                this._canvasModel.set("zoomLevel", this.canvas.getZoom());
            }
        },

        /**
         * Clear the `WorkflowModel`'s selected view if it is equal to the given view.
         *
         * @param {Backbone.View} view The view.
         * @private
         */
        _clearSelectedView: function (view) {
            var isSelected = this._canvasModel.get("selectedView") === view;
            isSelected && this._canvasModel.selectView(null);
        },

        /**
         * Create a Draw2D canvas.
         *
         * @param {element} container The container to show the canvas in.
         * @private
         * @return {JIRA.WorkflowDesigner.Draw2DCanvas} The newly created canvas.
         */
        _createCanvas: function (container) {
            var canvas,
                getBoundingBox = _.bind(this.getCanvasBoundingBox, this),
                id = this._getUniqueId(),
                layers;

            container = jQuery(container);

            // We must set the element's ID so Raphael can find it.
            container.append(this.el);
            this.$el.attr("id", id);

            // Figure layers from back to front.
            layers = [
                "statuses",
                "selected-status",
                "transitions",
                "global-transitions",
                "highlighted-transition",
                "selected-transition",
                "transition-labels"
            ];

            canvas = new JIRA.WorkflowDesigner.Draw2DCanvas(id);
            canvas.createLayers(layers);
            canvas.installEditPolicy(new JIRA.WorkflowDesigner.Policy.Canvas.PanningSingleSelectionPolicy(getBoundingBox));
            canvas.installEditPolicy(new JIRA.WorkflowDesigner.Policy.Canvas.SnapToGeometry.SnapEditPolicy());
            canvas.installEditPolicy(new JIRA.WorkflowDesigner.Policy.Canvas.SnapToGeometry.GuideLineEditPolicy());
            canvas.onNewConnection = _.bind(this._onNewConnection, this);
            return canvas;
        },

        /**
         * Delete the currently selected view.
         *
         * @return {boolean} Whether anything was deleted.
         * @private
         */
        _delete: function () {
            var selectedView = this._canvasModel.get("selectedView");
            selectedView && selectedView.destroy();
            return !!selectedView;
        },

        /**
         * @returns {JIRA.WorkflowDesigner.StatusPort[]} All possible target ports for any given port.
         * @private
         */
        _getAllTargetPorts: function () {
            function isInitial(statusView) {
                return statusView.model.isInitial();
            }

            function getPorts(statusView) {
                return statusView.getPorts();
            }

            return this.statusViews.chain().reject(isInitial).flatMap(getPorts).value();
        },

        /**
         * Calculate the minimum bounding box containing relevant items on the canvas.
         *
         * @return {draw2d.geo.Rectangle|undefined} The minimum bounding box containing relevant items on the canvas.
         */
        getCanvasBoundingBox: function () {
            var canvasBoundingBox,
                figures = this.canvas.getFigures().asArray(),
                lines = this.canvas.getLines().asArray();

            function addFigureBoundingBox(figure) {
                var boundingBox = figure.getBoundingBox();

                if (canvasBoundingBox) {
                    canvasBoundingBox = canvasBoundingBox.union(boundingBox);
                } else {
                    canvasBoundingBox = boundingBox;
                }
            }

            function isIgnored(figure) {
                return figure instanceof JIRA.WorkflowDesigner.Draw2DCanvas.LayerRootFigure;
            }

            _.chain(figures).reject(isIgnored).each(addFigureBoundingBox);
            _.each(lines, addFigureBoundingBox);
            return canvasBoundingBox;
        },

        /**
         * @param {JIRA.WorkflowDesigner.TransitionView} transitionView A transition view.
         * @private
         * @return {JIRA.WorkflowDesigner.TransitionView[]} `transitionView`'s sibling (common) transitions.
         */
        _getSiblingTransitions: function (transitionView) {
            var actionId = transitionView.model.get("actionId");

            function isSibling(transitionView) {
                return transitionView.model.get("actionId") === actionId;
            }

            return this.transitionViews.chain().filter(isSibling).without(transitionView).value();
        },

        /**
         * Get the view corresponding to the given status model.
         *
         * @param {JIRA.WorkflowDesigner.StatusModel} statusModel The status model.
         * @return {JIRA.WorkflowDesigner.StatusView|undefined} The view corresponding to <tt>statusModel</tt>.
         */
        _getStatusViewWithModel: function (statusModel) {
            return this.statusViews.find(function (statusView) {
                return statusView.model === statusModel;
            });
        },

        /**
         * Get the view with the given port.
         *
         * @param {draw2d.Port} port The port.
         * @return {JIRA.WorkflowDesigner.StatusView|undefined} The <tt>StatusView</tt> associated with <tt>port</tt>.
         * @private
         */
        getStatusViewWithPort: function (port) {
            return this.statusViews.find(function (statusView) {
                return _.contains(statusView.getPorts(), port);
            });
        },

        /**
         * Get the view corresponding to the given transition model.
         *
         * @param {JIRA.WorkflowDesigner.TransitionModel} transitionModel The transition model.
         * @return {JIRA.WorkflowDesigner.TransitionView|undefined} The view corresponding to <tt>transitionModel</tt>.
         */
        _getTransitionViewWithModel: function (transitionModel) {
            return this.transitionViews.find(function (transitionView) {
                return transitionView.model === transitionModel;
            });
        },

        /**
         * Generate a unique element ID.
         *
         * @return {string} A unique ID.
         * @private
         */
        _getUniqueId: function () {
            var i = 0,
                id;

            do {
                i++;
                id = "workflow-designer" + i;
            } while (jQuery("#" + id).length);

            return id;
        },

        /**
         * @returns {boolean}
         * @private
         */
        _isPortDragged: function () {
            return !!this.portDragged;
        },

        /**
         * @param {jQuery.Event} e
         * @private
         */
        _onKeyDown: function (e) {
            if (this._shouldHandleDelete(e)) {
                this._delete() && e.preventDefault();
            }
        },

        /**
         * Respond to the creation of a new connection.
         *
         * @param {draw2d.Connection} connection The new connection.
         * @private
         */
        _onNewConnection: function (connection) {
            var sourcePort = connection.getSource(),
                sourceView = this.getStatusViewWithPort(sourcePort),
                targetPort = connection.getTarget(),
                targetView = this.getStatusViewWithPort(targetPort),
                transition;

            transition = new JIRA.WorkflowDesigner.TransitionModel({
                source: sourceView.model,
                sourceAngle: sourceView.getAngleToPort(sourcePort),
                target: targetView.model,
                targetAngle: targetView.getAngleToPort(targetPort)
            });

            this.canvas.removeFigure(connection);
            new JIRA.WorkflowDesigner.Dialogs.AddTransitionDialogView({
                canvasModel: this._canvasModel,
                transitionModel: transition,
                workflowModel: this._workflowModel
            }).show();
        },

        /**
         * Respond to the re-connecting of a transition.
         *
         * Called when a connection is dragged to a different status or on redo.
         *
         * @param {JIRA.WorkflowDesigner.TransitionView} transitionView The transition that was reconnected.
         * @private
         */
        _onReconnect: function (transitionView) {
            var sourceChanged,
                sourcePort = transitionView.getConnection().getSource(),
                sourceView = this.getStatusViewWithPort(sourcePort),
                targetChanged,
                targetPort = transitionView.getConnection().getTarget(),
                targetView = this.getStatusViewWithPort(targetPort);

            sourceChanged = sourceView.model !== transitionView.model.get("source");
            targetChanged = targetView.model !== transitionView.model.get("target");

            if (sourceChanged) {
                new JIRA.WorkflowDesigner.Dialogs.EditTransitionSourceDialogView({
                    newSourcePort: sourcePort,
                    newSourceView: sourceView,
                    originalSourceStatus: transitionView.model.get("source"),
                    transitionView: transitionView,
                    workflowModel: this._workflowModel
                }).show();
            } else if (targetChanged) {
                new JIRA.WorkflowDesigner.Dialogs.EditTransitionTargetDialogView({
                    targetPort: targetPort,
                    targetView: targetView,
                    transitionView: transitionView,
                    workflowModel: this._workflowModel
                }).show();
            } else {
                transitionView.model.set({
                    source: sourceView.model,
                    sourceAngle: sourceView.getAngleToPort(sourcePort),
                    target: targetView.model,
                    targetAngle: targetView.getAngleToPort(targetPort)
                });
            }
        },

        /**
         * Deselect a transition and all its siblings.
         *
         * @param {JIRA.WorkflowDesigner.TransitionView} transitionView A transition view.
         * @private
         */
        _onTransitionDeselected: function (transitionView) {
            this._clearSelectedView(transitionView);
            _.invoke(this._getSiblingTransitions(transitionView), "unhighlight");
        },

        /**
         * Select a transition and all its siblings.
         *
         * @param {JIRA.WorkflowDesigner.TransitionView} transitionView A transition view.
         * @private
         */
        _onTransitionSelected: function (transitionView) {
            this._setSelectedView(transitionView);
            _.invoke(this._getSiblingTransitions(transitionView), "appearSelected");
        },

        /**
         * Adjust the zoom level in response to a zoom gesture.
         *
         * @param {object} e The zoom event.
         * @private
         */
        _onZoom: function (e) {
            AJS.InlineDialog.current && AJS.InlineDialog.current.hide();
            this.zoom(e.factor, this.canvas.fromDocumentToCanvasCoordinate(e.clientX, e.clientY));
        },

        /**
         * Position statuses that don't yet have coordinates.
         *
         * @private
         */
        _positionNewStatuses: function () {
            var statusViews = this._workflowModel.get("statuses").map(this._getStatusViewWithModel, this);
            JIRA.WorkflowDesigner.StatusPositioner.positionStatuses(statusViews, this.canvas.getViewBox());
        },

        /**
         * Destroy the view.
         */
        remove: function () {
            Backbone.View.prototype.remove.apply(this, arguments);
            jQuery(document).off("keydown", this._onKeyDown);
            this._zoomHandler.destroy();
        },

        /**
         * Remove a status from the canvas.
         *
         * @param {JIRA.WorkflowDesigner.StatusModel} statusModel The status to be removed.
         */
        removeStatus: function (statusModel) {
            var statusView = this._getStatusViewWithModel(statusModel);

            if (statusView) {
                statusView.remove();
                this.statusViews.remove(statusView);
            }
        },

        /**
         * Remove a transition from the canvas.
         *
         * @param {JIRA.WorkflowDesigner.TransitionModel} transitionModel The transition to be removed.
         */
        removeTransition: function (transitionModel) {
            var transitionView = this._getTransitionViewWithModel(transitionModel);

            if (transitionView) {
                transitionView.remove();
                this.transitionViews.remove(transitionView);
            }
        },

        /**
         * Render the canvas.
         *
         * @param {element} container The element to show the canvas in.
         * @return {JIRA.WorkflowDesigner.CanvasView} `this`
         */
        render: function (container) {
            this.canvas || (this.canvas = this._createCanvas(container));
            return this;
        },

        /**
         * Reset the canvas statuses to match those of the model.
         */
        resetStatuses: function () {
            while (this.statusViews.length) {
                this.removeStatus(this.statusViews.at(0).model);
            }

            this._workflowModel.get("statuses").each(function (statusModel) {
                this.addStatus(statusModel);
            }, this);
        },

        /**
         * Reset the canvas transitions to match those of the model.
         */
        resetTransitions: function () {
            while (this.transitionViews.length) {
                this.removeTransition(this.transitionViews.at(0).model);
            }

            this._workflowModel.get("transitions").forEach(function (transitionModel) {
                this.addTransition(transitionModel);
            }, this);
        },

        /**
         * Determine whether a transition should be considered selected.
         *
         * @param {JIRA.WorkflowDesigner.TransitionView} transitionView A transition view.
         * @private
         * @return {boolean} Whether `transitionView` should be considered selected.
         */
        _transitionIsSelected: function (transitionView) {
            var actionId = transitionView.model.get("actionId"),
                isSelected = this._canvasModel.get("selectedView") === transitionView,
                selectedModel = this._canvasModel.get("selectedModel"),
                siblingIsSelected = !!selectedModel && selectedModel.get("actionId") === actionId;

            return isSelected || siblingIsSelected;
        },

        /**
         * Select the view corresponding to a given model.
         *
         * @param {JIRA.WorkflowDesigner.CanvasModel} canvasModel The application's `CanvasModel`.
         * @param {Backbone.Model} selectedModel A model.
         * @private
         */
        _onSelectedModelChange: function (canvasModel, selectedModel) {
            var selectedModelView = this._getStatusViewWithModel(selectedModel) || this._getTransitionViewWithModel(selectedModel),
                selectedView = this._canvasModel.get("selectedView");

            // If the selected model has a view, ensure it is selected.
            if (selectedModel && selectedModelView && selectedModelView !== selectedView) {
                selectedModelView.select();
            }
        },

        /**
         * Handler for port drag end
         * @private
         */
        _onPortDragEnd: function () {
            function isInitial(statusView) {
                return statusView.model.isInitial();
            }

            this._setPortDragged(false);
            this.statusViews.chain().reject(isInitial).invoke("updatePortsVisibility");
        },

        /**
         * Lock or unlock the view, preventing or enabling interaction.
         *
         * @param {boolean} isLocked Whether the view should be locked.
         */
        setLocked: function (isLocked) {
            this._isLocked = isLocked;
        },

        /**
         * Sets the port being dragged
         * @param {draw2d.Port} portDragged
         * @private
         */
        _setPortDragged: function (portDragged) {
            this.portDragged = portDragged;
        },

        /**
         * Set how often the canvas resizes to fill its container.
         *
         * @param {number} resizeInterval The new resize interval (in milliseconds).
         */
        setResizeInterval: function (resizeInterval) {
            this.canvas.setResizeInterval(resizeInterval);
        },

        /**
         * Set the WorkflowModel's selected view.
         *
         * @param {Backbone.View} view The view.
         * @private
         */
        _setSelectedView: function (view) {
            this._canvasModel.selectView(view);
        },

        /**
         * @param {object} e A key down event.
         * @return {boolean} Whether the delete request should be handled.
         * @private
         */
        _shouldHandleDelete: function (e) {
            var deleteKeyCodes = [jQuery.ui.keyCode.BACKSPACE, jQuery.ui.keyCode.DELETE],
                dialogIsVisible = !!jQuery(".aui-blanket:visible").length,
                isDelete = _.contains(deleteKeyCodes, e.which),
                isInput = jQuery(e.target).is(":input");

            return !dialogIsVisible && isDelete && !isInput && !this._isLocked;
        },

        /**
         * Updates the transition view to match the model.
         * @private
         */
        _updateTransitionSourceAndTarget: function(transitionModel) {
            var sourceView = this._getStatusViewWithModel(transitionModel.get("source")),
                targetView = this._getStatusViewWithModel(transitionModel.get("target")),
                transitionView = this._getTransitionViewWithModel(transitionModel);

            transitionView.setViews(sourceView, targetView);
            transitionView.resetConnection();
        },

        /**
         * Zoom in/out on a target point.
         *
         * @param {number} factor The zoom factor.
         * @param {draw2d.geo.Point} [target] The target point (defaults to the centre of the canvas).
         */
        zoom: function (factor, target) {
            this.canvas.zoom(factor, target || this.canvas.getViewBox().getCenter(), this.getCanvasBoundingBox());
            this._canvasModel.set("zoomLevel", this.canvas.getZoom());
        }
    }));
}(JIRA.WorkflowDesigner.Backbone));