(function (Backbone) {
    Backbone.define("JIRA.WorkflowDesigner.WorkflowModel", Backbone.Model.extend(
    /** @lends JIRA.WorkflowDesigner.WorkflowModel# */
    {
        defaults: function () {
            return {
                currentStepId: null,
                draft: false,
                name: null,
                liveStepIds: null,
                loadedAt: null,
                statuses: new JIRA.WorkflowDesigner.StatusCollection(),
                transitions: new JIRA.WorkflowDesigner.TransitionCollection(),
                updatedDate: null,
                updateAuthor: null
            };
        },

        /**
         * @classdesc A JIRA workflow, containing statuses and transitions.
         * @constructs
         * @extends Backbone.Model
         */
        initialize: function () {
            var triggerLayoutChanged = _.bind(this.trigger, this, "layoutChanged");

            this.listenTo(this.get("statuses"), {
                "change:x change:y": triggerLayoutChanged,
                reset: this._detectNewStatus
            });

            this.listenTo(this.get("transitions"), {
                "change:actionId change:sourceAngle change:target change:targetAngle remove": triggerLayoutChanged,
                reset: this._detectNewTransition
            });
        },

        /**
         * Add a status to the workflow.
         *
         * @param {JIRA.WorkflowDesigner.StatusModel|object} status The status or its attributes.
         * @return {JIRA.WorkflowDesigner.StatusModel} The new status.
         */
        addStatus: function (status) {
            var isStatus = status instanceof JIRA.WorkflowDesigner.StatusModel;
            isStatus || (status = new JIRA.WorkflowDesigner.StatusModel(status));
            this.get("statuses").add(status);
            return status;
        },

        /**
         * Add a transition to the workflow.
         *
         * @param {JIRA.WorkflowDesigner.TransitionModel|object} transition The transition or its attributes.
         * @return {JIRA.WorkflowDesigner.TransitionModel} The new transition.
         */
        addTransition: function (transition) {
            var isTransition = transition instanceof JIRA.WorkflowDesigner.TransitionModel;
            isTransition || (transition = new JIRA.WorkflowDesigner.TransitionModel(transition));
            this.get("transitions").add(transition);
            return transition;
        },

        /**
         * Trigger a "new:status" event if a single new status is detected.
         *
         * Called when the status collection is reset.
         *
         * @param {JIRA.WorkflowDesigner.StatusCollection} statuses The status collection.
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.StatusModel[]} options.previousModels The previous contents of the collection.
         * @private
         */
        _detectNewStatus: function (statuses, options) {
            var newStatus = this._getNewModel(statuses, options.previousModels);
            newStatus && this.trigger("new:status", newStatus);
        },

        /**
         * Trigger a "new:transition" event if a single new transition is detected.
         *
         * Called when the transition collection is reset.
         *
         * @param {JIRA.WorkflowDesigner.StatusCollection} transitions The transition collection.
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.StatusModel[]} options.previousModels The previous contents of the collection.
         * @private
         */
        _detectNewTransition: function (transitions, options) {
            var newTransition = this._getNewModel(transitions, options.previousModels);
            newTransition && this.trigger("new:transition", newTransition);
        },

        /**
         * @param {JIRA.WorkflowDesigner.StatusModel} status A status.
         * @returns {JIRA.WorkflowDesigner.TransitionModel|undefined} The global
         *     transition associated with `status` or `undefined`.
         */
        getGlobalTransitionForStatus: function (status) {
            return this.get("transitions").findWhere({
                globalTransition: true,
                target: status
            });
        },

        /**
         * Get the one new model in an array of models.
         *
         * @param {Backbone.Collection} newModels The new models.
         * @param {Backbone.Model[]} oldModels The old models.
         * @return {Backbone.Model|undefined} The one new model or undefined.
         * @private
         */
        _getNewModel: function (newModels, oldModels) {
            var isInitialLoad = oldModels.length === 0;

            newModels = newModels.filter(function (newModel) {
                return _.every(oldModels, function (oldModel) {
                    return newModel.id !== oldModel.id;
                });
            });

            if (newModels.length === 1 && !isInitialLoad) {
                return newModels[0];
            }
        },

        /**
         * @param {string} id The ID of the status to retrieve.
         * @return {JIRA.WorkflowDesigner.StatusModel|undefined} The status with
         *     the given ID or `undefined`.
         */
        getStatus: function (id) {
            return this.get("statuses").get(id);
        },

        /**
         * Removes a transition model from this workflow model.
         *
         * @param {JIRA.WorkflowDesigner.TransitionModel} transition Transition model to remove.
         */
        removeTransition: function (transition) {
            this.get("transitions").remove(transition);
        },

        /**
         * Reset the model to a given state.
         *
         * Triggers "reset:before" and "reset:after" events.
         *
         * @param {object} attributes
         * @param {boolean} [attributes.draft] Whether the workflow is a draft.
         * @param {string} [attributes.name] The workflow's name.
         * @param {JIRA.WorkflowDesigner.StatusModel[]} attributes.statuses The workflow's statuses.
         * @param {JIRA.WorkflowDesigner.TransitionModel[]} attributes.transitions The workflow's transitions.
         */
        reset: function (attributes) {
            this.trigger("reset:before");
            this.get("statuses").reset(attributes.statuses);
            this.get("transitions").reset(attributes.transitions);
            this.set(_.omit(attributes, "statuses", "transitions"));
            this.trigger("reset:after");
        },

        /**
         * @param {JIRA.WorkflowDesigner.StatusModel} status A status.
         * @returns {boolean} Whether `status` is the target of a global transition.
         */
        statusHasGlobalTransition: function (status) {
            return !!this.getGlobalTransitionForStatus(status);
        },

        /**
         * A status can't be deleted if it's a draft workflow and the same status exists on the live workflow.
         *
         * @param {JIRA.WorkflowDesigner.StatusModel} status Status model.
         * @returns {boolean} Whether the given status can be deleted.
         */
        statusIsDeletable: function (status) {
            var isDraft = this.get("draft"),
                isLiveStatus = _.contains(this.get("liveStepIds"), status.get("stepId"));

            return !(isDraft && isLiveStatus);
        },

        /**
         * Updates the source step of the specified transition.
         *
         * @param {object} options
         * @param {number} options.sourceAngle Source angle.
         * @param {number} options.sourceStepId The ID of the transition's source step.
         * @param {JIRA.WorkflowDesigner.TransitionModel} options.transition The transition.
         */
        updateTransitionSource: function(options) {
            var sourceStatus = this.get("statuses").findWhere({
                initial: false,
                stepId: options.sourceStepId
            });

            options.transition.setSource(sourceStatus, options.sourceAngle);
        },

        /**
         * Updates the target step of the specified transition. Will update multiple transition models in case the transition is common.
         *
         * @param {object} options
         * @param {number} options.targetAngle Target angle.
         * @param {number} options.targetStepId The ID of the transition's target step.
         * @param {JIRA.WorkflowDesigner.TransitionModel} options.transition The transition.
         */
        updateTransitionTargets: function(options) {
            var actionId = options.transition.get("actionId"),
                targetStatus;

            targetStatus = this.get("statuses").findWhere({
                initial: false,
                stepId: options.targetStepId
            });

            options.transition.setTarget(targetStatus, options.targetAngle);

            // Other connections of the same common transition.
            this.get("transitions").chain()
                .filter(function(transition) { return transition !== options.transition && transition.get("actionId") === actionId; })
                .invoke("setTarget", targetStatus);
        }
    }));
}(JIRA.WorkflowDesigner.Backbone));