(function (Backbone) {

    Backbone.define("JIRA.WorkflowDesigner.CanvasModel", Backbone.Model.extend(
    /** @lends JIRA.WorkflowDesigner.CanvasModel# */
    {
        defaults: {
            selectedModel: null,
            selectedView: null,
            showTransitionLabels: false,
            zoomLevel: null
        },

        /**
         * @classdesc The state of the workflow designer's UI.
         * @constructs
         * @extends Backbone.Model
         * @param {object} attributes
         * @param {object} options
         * @param {JIRA.WorkflowDesigner.WorkflowModel} options.workflowModel The application's workflow model.
         */
        initialize: function (attributes, options) {
            this._workflowModel = options.workflowModel;

            this.listenTo(this, "change:showTransitionLabels", this._onShowTransitionLabelsChange);
            this.listenTo(this._workflowModel, {
                "new:status": this._selectModel,
                "new:transition": this._selectModel,
                "reset:before": this._deselect
            });

            this.listenTo(this._workflowModel.get("transitions"), "destroy", this._onDestroy);
        },

        /**
         * Deselect the currently selected object.
         *
         * @private
         */
        _deselect: function () {
            this.set({
                selectedModel: null,
                selectedView: null
            });
        },

        /**
         * Deselect the selected model/view if it was destroyed.
         *
         * @param {Backbone.Model} model The model that was destroyed.
         * @private
         */
        _onDestroy: function (model) {
            var isSelected = this.get("selectedModel") === model;
            isSelected && this._deselect();
        },

        /**
         * Persist the user's "show transition labels" preference.
         *
         * @param {JIRA.WorkflowDesigner.CanvasModel} model `this`
         * @param {boolean} value The user's "show transition labels" preference.
         * @private
         */
        _onShowTransitionLabelsChange: function (model, value) {
            JIRA.WorkflowDesigner.Analytics.triggerToggleLabels(value);
        },

        /**
         * Set the selected model if it isn't already set.
         *
         * @param {Backbone.Model} model The model to select.
         * @private
         */
        _selectModel: function (model) {
            this.get("selectedModel") || this.set("selectedModel", model);
        },

        /**
         * Set the model's selected model and view from a view.
         *
         * @param {Backbone.View} view The view.
         */
        selectView: function (view) {
            this.set({
                selectedModel: view && view.model,
                selectedView : view
            });
        }
    }));
}(JIRA.WorkflowDesigner.Backbone));