package com.atlassian.integrationtesting.ui;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import com.atlassian.integrationtesting.ui.CompositeUiTester.Backup;
import com.atlassian.integrationtesting.ui.CompositeUiTester.Login;
import com.atlassian.integrationtesting.ui.CompositeUiTester.WebSudoLogin;
import com.atlassian.integrationtesting.ui.UiTesters.BackupFile;

import com.gargoylesoftware.htmlunit.html.HtmlPage;
import com.google.common.base.Function;

import org.apache.commons.io.output.ByteArrayOutputStream;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import be.roam.hue.doj.Doj;

import static com.atlassian.integrationtesting.ui.UiTesters.doNothingWebSudoLogin;
import static com.atlassian.integrationtesting.ui.UiTesters.getHome;
import static com.atlassian.integrationtesting.ui.UiTesters.goToPageToLogout;
import static com.atlassian.integrationtesting.ui.UiTesters.isOnLogInPageByFormId;
import static com.atlassian.integrationtesting.ui.UiTesters.logInByGoingTo;
import static org.apache.commons.io.FileUtils.forceMkdir;
import static org.apache.commons.io.IOUtils.closeQuietly;
import static org.apache.commons.io.IOUtils.copy;
import static org.apache.commons.lang.RandomStringUtils.randomAlphanumeric;

/**
 * {@code UiTesterProvider}s for Bamboo versions 2.5, 2.6 and 3.0.
 */
public enum Bamboo implements UiTesterFunctionProvider
{
    v2_5,
    v2_6,
    v3_0;
    
    private final Function<Login, HtmlPage> logIn = logInByGoingTo("userlogin!default.action").formId("loginForm").build();
    public Function<Login, HtmlPage> logIn()
    {
        return logIn;
    }
    
    public Function<WebSudoLogin, HtmlPage> webSudoLogIn()
    {
        return doNothingWebSudoLogin();
    }

    private final Function<UiTester, Boolean> isOnLogInPage = isOnLogInPageByFormId("loginForm");
    public Function<UiTester, Boolean> isOnLogInPage()
    {
        return isOnLogInPage;
    }

    public Function<UiTester, Void> logout()
    {
        return goToPageToLogout("userLogout.action");
    }

    public Function<UiTester, String> getLoggedInUser()
    {
        return UiTesters.getLoggedInUser();
    }
    
    public Function<Backup, Void> restore()
    {
        return UiTesters.restore(ProcessBackupData.INSTANCE, Restore.INSTANCE);
    }
    
    private enum Restore implements Function<BackupFile, Void>
    {
        INSTANCE;

        public Void apply(BackupFile backup)
        {
            try
            {
                backup.client.gotoPage("admin/import!default.action");
                // fill in file path
                backup.client.currentPage().getById("importForm_path").value(backup.file.getAbsolutePath());
                // don't bother backing up existing data
                backup.client.currentPage().getById("importForm_backupSelected").uncheck();
                // apply restore without shutdown
                backup.client.currentPage().getById("importForm_hotSwapSelected").check();
                // submit
                HtmlPage page = (HtmlPage) backup.client.currentPage().get(".formFooter input").attribute("type", "submit").click();
                // confirm
                Doj.on(page).get(".formFooter input").attribute("type", "submit").click();
            }
            catch (ClassCastException e)
            {
                throw new RestoreFromBackupException("Element with id 'restore_submit' is not a button");
            }
            catch (IOException e)
            {
                throw new RestoreFromBackupException(e);
            }
            return null;
        }
    }
    
    private enum ProcessBackupData implements Function<Backup, File>
    {
        INSTANCE;
        
        private static final String ADMIN_CONFIG = "configuration/administration.xml";
        
        public File apply(Backup backup)
        {
            File dataFile = new File(getHome(), "import/backup" + randomAlphanumeric(5) + ".zip");
            
            InputStream in = null;
            ZipInputStream zin = null;
            
            OutputStream out = null; 
            ZipOutputStream zout = null;
            
            try
            {
                forceMkdir(dataFile.getParentFile());
                dataFile.createNewFile();

                in = backup.data.openStream();
                zin = new ZipInputStream(in);
                
                out = new FileOutputStream(dataFile);
                zout = new ZipOutputStream(out);
                
                ZipEntry inEntry = null;
                while ((inEntry = zin.getNextEntry()) != null)
                {
                    if (inEntry.getName().equals(ADMIN_CONFIG))
                    {
                        ByteArrayOutputStream baos = new ByteArrayOutputStream();
                        copy(zin, baos);
                        ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());

                        Document doc = parse(bais);
                        updateBaseUrl(doc, backup.client.getBaseUrl());
                        
                        zout.putNextEntry(new ZipEntry(ADMIN_CONFIG));
                        write(doc, zout);
                    }
                    else
                    {
                        zout.putNextEntry(new ZipEntry(inEntry.getName()));
                        copy(zin, zout);
                        zout.closeEntry();
                    }
                }
            }
            catch (IOException e)
            {
                throw new RestoreFromBackupException("Unable to process backup data file " + backup.data, e);
            }
            catch (ParserConfigurationException e)
            {
                throw new RestoreFromBackupException("Unable to process backup data file " + backup.data, e);
            }
            catch (SAXException e)
            {
                throw new RestoreFromBackupException("Unable to process backup data file " + backup.data, e);
            }
            catch (TransformerFactoryConfigurationError e)
            {
                throw new RestoreFromBackupException("Unable to process backup data file " + backup.data, e);
            }
            catch (TransformerException e)
            {
                throw new RestoreFromBackupException("Unable to process backup data file " + backup.data, e);
            }
            finally
            {
                closeQuietly(zin);
                closeQuietly(in);
                closeQuietly(zout);
                closeQuietly(out);
            }
            return dataFile;
        }

        private void write(Document doc, OutputStream out) throws TransformerFactoryConfigurationError, TransformerException
        {
            Transformer transformer = TransformerFactory.newInstance().newTransformer();
            StreamResult result = new StreamResult(out);
            DOMSource source = new DOMSource(doc);
            transformer.transform(source, result);
        }

        private void updateBaseUrl(Document doc, String baseUrl)
        {
            updateElementWithText(doc, "myBaseUrl", baseUrl);
        }
        
        private void updateElementWithText(Document doc, String elementName, String value)
        {
            NodeList nodes = doc.getElementsByTagName(elementName);
            if (nodes.getLength() != 0)
            {
                nodes.item(0).getParentNode().removeChild(nodes.item(0));
            }
            Element element = doc.createElement(elementName);
            element.appendChild(doc.createTextNode(value));
            doc.getDocumentElement().appendChild(element);
        }

        private Document parse(ByteArrayInputStream bais) throws ParserConfigurationException, SAXException, IOException
        {
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder db = dbf.newDocumentBuilder();
            return db.parse(bais);
        }
    }
}
