package com.atlassian.crowd.directory.synchronisation.cache;

import com.atlassian.crowd.directory.synchronisation.utils.AddUpdateSets;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithCredentialAndAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * A cache of users, groups and memberships for an external Directory.
 *
 * The {@link CacheRefresher} passes the updated data to this interface for caching.
 *
 * @see UsnChangedCacheRefresher
 */
public interface DirectoryCache {
    // -----------------------------------------------------------------------------------------------------------------
    // Batch operations
    // -----------------------------------------------------------------------------------------------------------------

    /**
     * Synchronises a list of Users from an external directory in the local cache.
     * <p>
     * If the syncStartDate is not null, then it is used to ensure we don't re-introduce stale data if a User is updated
     * locally after we did our search.
     * Some callers may intentionally choose to pass null - eg  when {@link com.atlassian.crowd.directory.ldap.cache.UsnChangedCacheRefresher} does a partial synchronise.
     *
     * @param users         A list of Users from the external directory.
     * @param syncStartDate The date that the synchronise started (can be null).
     * @throws com.atlassian.crowd.exception.OperationFailedException If the Internal Directory throws a OperationFailedException - which seems unlikely.
     */
    AddUpdateSets<UserTemplateWithCredentialAndAttributes, UserTemplate> addOrUpdateCachedUsers(Collection<? extends User> users, Date syncStartDate) throws OperationFailedException;

    void deleteCachedUsersNotIn(Collection<? extends User> users, Date syncStartDate) throws OperationFailedException;

    void deleteCachedUsersByGuid(Set<String> guids) throws OperationFailedException;

    void addOrUpdateCachedGroups(Collection<? extends Group> groups, Date syncStartDate) throws OperationFailedException;

    void deleteCachedGroupsNotIn(GroupType groupType, List<? extends Group> remoteGroups, Date syncStartDate) throws OperationFailedException;

    void deleteCachedGroupsNotInByExternalId(final Collection<? extends Group> remoteGroups, Date syncStartDate) throws OperationFailedException;

    void deleteCachedGroups(Set<String> groupnames) throws OperationFailedException;

    void deleteCachedGroupsByGuids(Set<String> guids) throws OperationFailedException;

    void syncUserMembersForGroup(Group parentGroup, Collection<String> remoteUsers) throws OperationFailedException;

    void addUserMembersForGroup(Group parentGroup, Set<String> remoteUsers) throws OperationFailedException;

    void deleteUserMembersForGroup(Group parentGroup, Set<String> remoteUsers) throws OperationFailedException;

    void syncGroupMembersForGroup(Group parentGroup, Collection<String> groups) throws OperationFailedException;

    void addGroupMembersForGroup(Group parentGroup, Set<String> groups) throws OperationFailedException;

    void deleteGroupMembersForGroup(Group parentGroup, Set<String> groups) throws OperationFailedException;

    // -----------------------------------------------------------------------------------------------------------------
    // Event operations
    // -----------------------------------------------------------------------------------------------------------------

    void addOrUpdateCachedUser(User user) throws OperationFailedException;

    void deleteCachedUser(String username) throws OperationFailedException;

    void addOrUpdateCachedGroup(Group group) throws OperationFailedException;

    void deleteCachedGroup(String groupName) throws OperationFailedException;

    void addUserToGroup(String username, String groupName) throws OperationFailedException;

    void removeUserFromGroup(String username, String groupName) throws OperationFailedException;

    void addGroupToGroup(String childGroup, String parentGroup) throws OperationFailedException;

    void removeGroupFromGroup(String childGroup, String parentGroup) throws OperationFailedException;

    void syncGroupMembershipsForUser(String childUsername, Set<String> parentGroupNames) throws OperationFailedException;

    void syncGroupMembershipsAndMembersForGroup(String groupName, Set<String> parentGroupNames, Set<String> childGroupNames) throws OperationFailedException;

    /**
     * Returns a set of all user guids present in the cache excluding null values.
     *
     * @return set of all user guids present in the cache
     * @throws OperationFailedException if the internal directory throes a OperationFailedException
     */
    Set<String> getAllUserGuids() throws OperationFailedException;

    /**
     * Counts users in directory cache.
     *
     * @return user count
     */
    long getUserCount() throws OperationFailedException;

    UserWithAttributes findUserWithAttributesByName(String name) throws UserNotFoundException, OperationFailedException;

    /**
     * Searches the specified directory for user names of users with the specified external ids, returns
     * a map from external id to username
     *
     * @param externalIds the external ids of the users to search for
     */
    Map<String, String> findUsersByExternalIds(Set<String> externalIds);

    /**
     * Returns a set of all group guids present in the cache excluding null values.
     *
     * @return set of all group guids present in the cache
     * @throws OperationFailedException if the internal directory throes a OperationFailedException
     */
    Set<String> getAllGroupGuids() throws OperationFailedException;

    /**
     * Returns a set of all local group names.
     *
     * @return set of all local group names.
     * @throws OperationFailedException if the internal directory throes a OperationFailedException
     */
    Set<String> getAllLocalGroupNames() throws OperationFailedException;

    /**
     * Counts groups in directory cache.
     *
     * @return group count
     */
    long getGroupCount() throws OperationFailedException;

    /**
     * Counts external cached groups in the directory cache
     *
     * @return external cached group count
     * @throws OperationFailedException if the internal directory throws a OperationFailedException
     */
    long getExternalCachedGroupCount() throws OperationFailedException;

    /**
     * Searches the specified directory for group names of groups with the specified external ids
     *
     * @param externalIds external ids of the groups to find
     * @return a map from external ids to group names
     */
    Map<String, String> findGroupsByExternalIds(Set<String> externalIds) throws OperationFailedException;

    /**
     * Searches the specified directory for externalIds of groups with the specified names
     *
     * @param groupNames names of the groups to find
     * @return a map from group names to external ids
     */
    Map<String, String> findGroupsExternalIdsByNames(Set<String> groupNames) throws OperationFailedException;

    GroupWithAttributes findGroupWithAttributesByName(String name) throws GroupNotFoundException, OperationFailedException;

    /**
     * Removes any syncing user attributes listed in deletedAttributes, and store any that are listed in storedAttributes
     *
     * If any non-syncing attributes are included, they will be ignored
     *
     * @param userName          user to apply attributes to
     * @param deletedAttributes attribute keys that need to be deleted
     * @param storedAttributes  attributes and values that need to be saved (will overwrite any existing attributes with the same keys)
     */
    void applySyncingUserAttributes(String userName, Set<String> deletedAttributes, Map<String, Set<String>> storedAttributes)
            throws UserNotFoundException, OperationFailedException;

    /**
     * Removes any syncing group attributes listed in deletedAttributes, and store any that are listed in storedAttributes
     *
     * If any non-syncing attributes are included, they will be ignored
     *
     * @param groupName         group to apply attributes to
     * @param deletedAttributes attribute keys that need to be deleted
     * @param storedAttributes  attributes and values that need to be saved (will overwrite any existing attributes with the same keys)
     */
    void applySyncingGroupAttributes(String groupName, Set<String> deletedAttributes, Map<String, Set<String>> storedAttributes) throws GroupNotFoundException, OperationFailedException;
}
