package com.atlassian.crowd.audit;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;

public class AuditLogContextInternalImpl implements AuditLogContextInternal {

    private static final Logger log = LoggerFactory.getLogger(AuditLogContextInternalImpl.class);
    private final ThreadLocal<AuditLogAuthor> threadLocalAuthor;
    private final ThreadLocal<AuditLogEventSource> threadLocalSource;

    public AuditLogContextInternalImpl() {
        this(new ThreadLocal<>(), new ThreadLocal<>());
    }

    @VisibleForTesting
    AuditLogContextInternalImpl(ThreadLocal<AuditLogAuthor> threadLocalAuthor, ThreadLocal<AuditLogEventSource> threadLocalSource) {
        this.threadLocalAuthor = threadLocalAuthor;
        this.threadLocalSource = threadLocalSource;
    }

    @Override
    public Optional<AuditLogAuthor> getAuthor() {
        return Optional.ofNullable(threadLocalAuthor.get());
    }

    @Override
    public Optional<AuditLogEventSource> getSource() {
        return Optional.ofNullable(threadLocalSource.get());
    }

    @Override
    public <T> T withAuditLogAuthor(AuditLogAuthor author, AuditLogContextCallback<T> callback) throws Exception {
        try {
            log.debug("Setting custom author context for author {}", author);
            threadLocalAuthor.set(author);
            return callback.execute();
        } finally {
            threadLocalAuthor.remove();
            log.debug("Cleared custom author context for author {}", author);
        }
    }

    @Override
    public <T> T withAuditLogSource(AuditLogEventSource source, AuditLogContextCallback<T> callback) throws Exception {
        try {
            log.debug("Setting custom source context for source {}", source);
            threadLocalSource.set(source);
            return callback.execute();
        } finally {
            threadLocalSource.remove();
            log.debug("Cleared custom source context for source {}", source);
        }
    }
}
