package com.atlassian.crowd.search.util;

import com.atlassian.crowd.embedded.api.Query;
import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.model.NameComparator;
import org.apache.commons.lang3.tuple.Pair;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

/**
 * Utility class providing factory methods returning {@link ResultsAggregator}.
 */
public class ResultsAggregators {

    /**
     * Creates an instance that will use the provided function to uniquely identify results and that will
     * expect as many total results as indicated by the query.
     *
     * @param maker a key-making function
     * @param query an indication of how many results are required
     */
    public static <T, K extends Comparable<? super K>> ResultsAggregator<T> with(Function<? super T, ? extends K> maker, Query<? extends T> query) {
        return new ResultsAggregatorImpl<T, K>(maker, query);
    }

    public static <T, K extends Comparable<? super K>> ResultsAggregator<T> with(Function<? super T, ? extends K> maker,
                                                                                 int startIndex, int maxResults) {
        return new ResultsAggregatorImpl<T, K>(maker, startIndex, maxResults);
    }

    public static <T, K extends Comparable<? super K>> ResultsAggregator<T> with(Function<? super T, ? extends K> maker,
                                                                                 int startIndex, int maxResults, boolean merge) {
        if (merge) {
            return new ResultsAggregatorImpl<>(maker, startIndex, maxResults);
        } else {
            final AtomicInteger counter = new AtomicInteger();
            return new ResultsAggregatorImpl<>(e -> Pair.of(maker.apply(e), counter.incrementAndGet()), startIndex, maxResults);
        }
    }

    /**
     * Creates an aggregator that will identify and sort users by lower-cased name.
     *
     * @param startIndex defines how many results should be skipped
     * @param maxResults maximum number of results to be returned
     */
    public static <T extends User> ResultsAggregator<T> forUsers(int startIndex, int maxResults) {
        return with(user -> IdentifierUtils.toLowerCase(user.getName()), startIndex, maxResults);
    }

    /**
     * Creates an instance that will sort, de-duplicate and constrain results according to the {@code query}.
     */
    public static <T> ResultsAggregator<T> with(Query<T> query) {
        return with(query, true);
    }

    public static <T> ResultsAggregator<T> with(Query<T> query, boolean merge) {
        return with(NameComparator.normaliserOf(query.getReturnType()), query.getStartIndex(), query.getMaxResults(), merge);
    }

    public static <T> List<T> constrainResults(Query<T> query, Collection<T> values) {
        final ResultsAggregator<T> results = ResultsAggregators.with(query);
        results.addAll(values);
        return results.constrainResults();
    }
}
