package com.atlassian.crowd.manager.validation;

import java.net.InetAddress;

import javax.servlet.http.HttpServletRequest;

import com.atlassian.crowd.manager.proxy.TrustedProxyManager;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.util.I18nHelper;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Implements ClientValidationManager.
 */
public class ClientValidationManagerImpl implements ClientValidationManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(ClientValidationManagerImpl.class);
    private final TrustedProxyManager trustedProxyManager;
    private final I18nHelper i18nHelper;
    private final ApplicationRemoteAddressValidator applicationRemoteAddressValidator;

    public ClientValidationManagerImpl(final TrustedProxyManager trustedProxyManager,
                                       final I18nHelper i18nHelper,
                                       final ApplicationRemoteAddressValidator applicationRemoteAddressValidator) {
        this.trustedProxyManager = trustedProxyManager;
        this.i18nHelper = i18nHelper;
        this.applicationRemoteAddressValidator = checkNotNull(applicationRemoteAddressValidator);
    }

    @Override
    public void validate(final Application application, final HttpServletRequest request)
            throws ClientValidationException {
        Validate.notNull(application);
        Validate.notNull(request);

        validateApplicationActive(application);
        validateRemoteAddress(application, request);
    }

    private void validateApplicationActive(Application application) throws ClientValidationException {
        if (!application.isActive()) {
            throw new ClientValidationException(i18nHelper.getText("application.inactive.error", application.getName()));
        }
    }

    /**
     * Checks if the remote address is valid in the cache first before performing a more expensive remote address validation.
     *
     * @param application application to validate the remote address against
     * @param request     HTTP request
     * @throws ClientValidationException if the request address failed IP validation
     */
    private void validateRemoteAddress(final Application application, final HttpServletRequest request)
            throws ClientValidationException {
        final InetAddress clientAddress = XForwardedForUtil.getTrustedAddress(trustedProxyManager, request); // uses XFF if IP is a trusted proxy

        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Client address: " + clientAddress.getHostAddress());
        }

        if (!applicationRemoteAddressValidator.validate(application, clientAddress)) {
            LOGGER.info("Client with address '{}' is forbidden from making requests to application '{}'",
                    clientAddress.getHostAddress(), application.getName());
            String errorMsg = i18nHelper.getText("client.forbidden.exception", clientAddress.getHostAddress(), application.getName());

            throw new ClientValidationException(errorMsg);
        }
    }
}
