package com.atlassian.crowd.service.cluster;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.Internal;

import java.time.Instant;
import java.util.Optional;

public interface ClusterNode {
    /**
     * Retrieves a cluster-wide unique identifier for this node.
     * <p>
     * <b>Warning</b>: Plugins should make <i>no assumptions</i> about this ID. That includes:
     * <ul>
     * <li>How long it is</li>
     * <li>What characters it contains</li>
     * <li>Whether it will be consistent across restarts. Don't persist long-term data against these IDs</li>
     * </ul>
     * The value is guaranteed to be unique within the cluster. If the cluster is restarted, every node's ID will
     * still be unique but will likely not be the same as the previous value.
     *
     * @return a cluster-wide unique identifier for this node
     */
    String getNodeId();

    /**
     * A long living cluster name that is defined by setting the system property {@code cluster.node.name}
     *
     * @return The value from the system property named above otherwise an empty string if the property is not set.
     */
    String getNodeName();

    /**
     * @return the time when the node has last checked-in as available
     */
    @Internal
    Instant getLastHeartbeat();

    /**
     * @return true if this represents the node the method is executed on, false otherwise
     */
    boolean isLocal();

    /**
     * @return extra details about the cluster node, the presence and completeness of this data is
     * implementation specific, and not guaranteed
     */
    @ExperimentalApi
    Optional<ClusterNodeDetails> getDetails();
}