package com.atlassian.crowd.manager.mail;

import com.atlassian.crowd.manager.property.PropertyManagerException;
import com.atlassian.crowd.util.mail.SMTPServer;
import com.atlassian.crowd.validator.ValidationError;

import javax.activation.DataSource;
import javax.mail.internet.InternetAddress;
import java.util.Collection;
import java.util.List;
import java.util.Map;

public interface MailManager extends MailConfigurationService {
    /**
     * Sends an HTML email using the configured mail server (remote host / JNDI).
     *
     * @param emailAddress address of recipient.
     * @param subject      subject header.
     *                     Will be appended with the configured mail subject prefix in the final email.
     * @param body         email body html.
     * @param plainText    plainText alternative to the html body text specified in {@code body}
     * @throws MailSendException an error occured sending the email.
     */
    void sendHtmlEmail(InternetAddress emailAddress, String subject, String body, String plainText) throws MailSendException;

    /**
     * Sends an email using the configured mail server (remote host / JNDI).
     *
     * @param emailAddress address of recipient.
     * @param subject      subject header.
     *                     Will be appended with the configured mail subject prefix in the final email.
     * @param body         email body text.
     * @throws MailSendException an error occured sending the email.
     */
    void sendPlainTextEmail(InternetAddress emailAddress, String subject, String body) throws MailSendException;

    /**
     * Sends an email with zero or more attachments and headers using the configured mail server (remote host / JNDI).
     *
     * @param emailAddress address of recipient.
     * @param subject      subject header.
     *                     Will be appended with the configured mail subject prefix in the final email.
     * @param body         email body text.
     * @param headers      custom headers to add.
     * @param attachments  [displayed filename]-&gt;[data] mapping for attachments.
     * @throws MailSendException an error occured sending the email.
     * @since 2.11
     */
    void sendPlainTextEmail(InternetAddress emailAddress, String subject, String body,
                            Map<String, String> headers, Map<String, DataSource> attachments) throws MailSendException;

    /**
     * Send a test email using the supplied mail server (remote host / JNDI)
     *
     * @param smtpServer    supplied SMTP Server, which will be used to send message.
     * @param emailAddress  address of recipient.
     * @return result of test, which contains logs and status.
     * @throws MailSendException an error occured sending the email.
     * @since 3.2
     */
    SendTestMailResult sendTestMail(SMTPServer smtpServer, InternetAddress emailAddress) throws MailSendException;

    /**
     * Sends multiple emails using the configured mail server.
     *
     * @throws MailSendException an error occurred that prevented sending any email (connection error, configuration error, etc.).
     * @return a collection of messages that failed to be sent
     * @since 3.3
     */
    Collection<? extends EmailMessage> sendEmails(Collection<? extends EmailMessage> messagesToSend) throws MailSendException;

    /**
     * Verifies whether the specified MailConfiguration is valid
     *
     * @param mailConfiguration  entity with attributes to be validated
     * @return a list with errors found in the configuration
     * @since 3.2
     */
    List<ValidationError> validateConfiguration(MailConfiguration mailConfiguration);

    /**
     * Sets the new mail configuration
     *
     * @param mailServerEntity new configuration
     * @since 3.2
     * @deprecated Use {@link MailConfigurationService instead}
     */
    @Deprecated
    void saveConfiguration(MailConfiguration mailServerEntity);

    /**
     * Returns current mail configuration
     *
     * @throws PropertyManagerException property does no exist
     * @since 3.2
     * @deprecated Use {@link MailConfigurationService instead}
     */
    @Deprecated
    MailConfiguration getMailConfiguration();

    /**
     * @return <code>true</code> if the mail server is fully configured (does not check if the configuration is correct)
     * @deprecated Use {@link MailConfigurationService instead}
     */
    @Deprecated
    boolean isConfigured();
}
