package com.atlassian.crowd.service.token;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import com.atlassian.crowd.exception.ObjectAlreadyExistsException;
import com.atlassian.crowd.model.token.ExpirableUserToken;

/**
 * Service for handling invite user tokens
 *
 * @since v2.8
 */
public interface InviteUserTokenService {
    // The default expiry time should be long enough to be usable, but as short as possible so you don't have
    // valid tokens hanging around as possible attack vectors. As an 'invite user' token is not for immediate
    // consumption, 1 week seems reasonable.
    public static final int DEFAULT_TOKEN_EXPIRY_SECONDS = (int) TimeUnit.DAYS.toSeconds(7);

    /**
     * Create a new token for the given email address and directory ID and store it
     *
     * @param email              the email address that the invitation was sent to
     * @param directoryId        the directory ID that the user will be created to
     * @param tokenExpirySeconds number of seconds before generated token expires, or DEFAULT_TOKEN_EXPIRY_SECONDS
     * @return the created token
     * @throws ObjectAlreadyExistsException       if a token already exists with the same random hash as the one that was generated.
     * @throws java.lang.IllegalArgumentException if tokenExpirySeconds is less than 0
     */
    ExpirableUserToken createAndStoreToken(String email, long directoryId, int tokenExpirySeconds) throws ObjectAlreadyExistsException;

    /**
     * Find the token using the given random hash
     *
     * @param token random hash that was generated when the token was created
     * @return the token if it was found, none otherwise
     */
    Optional<ExpirableUserToken> findByToken(String token);

    /**
     * Removes the token with the given random hash
     *
     * @param token random hash that was generated when the token was created
     * @return true if a token was deleted, false otherwise
     */
    boolean removeToken(String token);
}
