package com.atlassian.crowd.service.client;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Locale;
import java.util.Properties;

import javax.annotation.Nullable;

import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.integration.Constants;
import com.atlassian.crowd.model.authentication.ApplicationAuthenticationContext;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.atlassian.crowd.integration.Constants.CROWD_SERVICE_LOCATION;
import static com.atlassian.crowd.integration.Constants.SECURITY_SERVER_NAME;
import static org.apache.commons.lang3.StringUtils.removeEnd;

/**
 * This bean is a container for the application's crowd.properties.
 */
public class ClientPropertiesImpl extends AbstractClientProperties {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    protected ClientPropertiesImpl() {
    }

    /**
     * Updates all the client properties with new vew values, which are read from the properties passed as argument.
     * All the properties are updated, not just the ones passed. The values are taken first from the system property,
     * if exists, then from the environment property, if exists, and then from the passed properties. If none of
     * those sources yields a value, then the value is set to null (for Strings) or 0L (for longs).
     *
     * @param properties properties to update from, unless overridden by system properties or environment variables
     */
    @Override
    public void updateProperties(Properties properties) {
        applicationName = loadAndLogPropertyString(properties, Constants.PROPERTIES_FILE_APPLICATION_NAME);
        applicationPassword = loadPropertyString(properties, Constants.PROPERTIES_FILE_APPLICATION_PASSWORD);
        applicationAuthenticationURL = loadAndLogPropertyString(properties, Constants.PROPERTIES_FILE_APPLICATION_LOGIN_URL);

        cookieTokenKey = loadPropertyString(properties, Constants.PROPERTIES_FILE_COOKIE_TOKENKEY);

        sessionTokenKey = loadAndLogPropertyString(properties, Constants.PROPERTIES_FILE_SESSIONKEY_TOKENKEY);
        sessionLastValidation = loadAndLogPropertyString(properties, Constants.PROPERTIES_FILE_SESSIONKEY_LASTVALIDATION);
        sessionValidationInterval = loadPropertyLong(properties, Constants.PROPERTIES_FILE_SESSIONKEY_VALIDATIONINTERVAL, true);

        httpProxyHost = loadPropertyString(properties, Constants.PROPERTIES_FILE_HTTP_PROXY_HOST);
        httpProxyPort = loadPropertyString(properties, Constants.PROPERTIES_FILE_HTTP_PROXY_PORT);
        httpProxyUsername = loadPropertyString(properties, Constants.PROPERTIES_FILE_HTTP_PROXY_USERNAME);
        httpProxyPassword = loadPropertyString(properties, Constants.PROPERTIES_FILE_HTTP_PROXY_PASSWORD);

        httpMaxConnections = loadPropertyString(properties, Constants.PROPERTIES_FILE_HTTP_MAX_CONNECTIONS);
        httpTimeout = loadPropertyString(properties, Constants.PROPERTIES_FILE_HTTP_TIMEOUT);
        socketTimeout = loadPropertyString(properties, Constants.PROPERTIES_FILE_SOCKET_TIMEOUT);
        ssoCookieDomainName = loadAndLogPropertyString(properties, Constants.PROPERTIES_FILE_COOKIE_DOMAIN);

        authenticationMethod = AuthenticationMethod.parse(
                StringUtils.defaultIfBlank(loadAndLogPropertyString(properties, Constants.AUTHENTICATION_METHOD),
                        AuthenticationMethod.BASIC_AUTH.getKey()));

        PasswordCredential credentials = new PasswordCredential(applicationPassword);
        applicationAuthenticationContext = new ApplicationAuthenticationContext();
        applicationAuthenticationContext.setName(applicationName);
        applicationAuthenticationContext.setCredential(credentials);

        baseURL = loadBaseURL(properties);
    }

    private long loadPropertyLong(Properties properties, String propertyName, boolean logProperty) {
        String propertyValueAsString;
        if (logProperty) {
            propertyValueAsString = loadAndLogPropertyString(properties, propertyName);
        } else {
            propertyValueAsString = loadPropertyString(properties, propertyName);
        }

        long propertyValue = 0L;
        if (propertyValueAsString != null) {
            propertyValue = Long.parseLong(propertyValueAsString);
        }

        return propertyValue;
    }

    @Nullable
    public String loadPropertyString(Properties properties, String propertyName) {
        String propertyValue = StringUtils.stripToNull(System.getProperty("crowd.property." + propertyName));

        if (propertyValue == null) {
            propertyValue = StringUtils.stripToNull(loadPropertyFromEnv(propertyName));
        }

        if (propertyValue == null && properties != null && properties.containsKey(propertyName)) {
            propertyValue = StringUtils.stripToNull(properties.getProperty(propertyName));
        }

        return propertyValue;
    }

    @Nullable
    private static String loadPropertyFromEnv(String propertyName) {
        // Convert property to environment variable name
        if (Boolean.getBoolean(Constants.USE_ENVIRONMENT_VARIABLES)) {
            String envPropertyName = "CROWD_PROPERTY_" + propertyName.toUpperCase(Locale.ENGLISH).replace(".", "_");
            return System.getenv(envPropertyName);
        }
        return null;
    }

    @Nullable
    protected String loadAndLogPropertyString(Properties properties, String propertyName) {
        String propertyValue = loadPropertyString(properties, propertyName);

        if (propertyValue != null) {
            logger.info("Loading property: '" + propertyName + "' : '" + propertyValue + "'");
        } else {
            logger.info("Failed to find value for property: " + propertyName);
        }

        return propertyValue;
    }

    @Nullable
    private String loadBaseURL(Properties properties) {
        String baseURL = loadPropertyString(properties, Constants.PROPERTIES_FILE_BASE_URL);
        if (StringUtils.isBlank(baseURL)) {
            baseURL = generateBaseURL(properties);
        }

        return StringUtils.removeEnd(baseURL, "/");
    }

    // We only want to trim /services from the path portion of the URI, not anywhere else.
    @Nullable
    private String generateBaseURL(Properties properties) {
        final String propertyUrl = loadPropertyString(properties, Constants.PROPERTIES_FILE_SECURITY_SERVER_URL);
        if (propertyUrl == null) {
            return null;
        }

        try {
            final URI uri = new URI(propertyUrl);
            final URI truncatedUri = new URI(uri.getScheme(), uri.getUserInfo(), uri.getHost(), uri.getPort(), truncatePath(uri.getPath()), uri.getQuery(), uri.getFragment());
            return truncatedUri.toString();
        } catch (URISyntaxException e) {
            return propertyUrl;
        }
    }

    /**
     * This method should truncate certain suffixes from path to get base service path.
     * Specifically, <code>/services</code> and <code>/services/SecurityServer</code> suffixes should be truncated
     * as well as trailing slash if exists. In case of any other suffixes original path should be returned unchanged.
     *
     * @param originalPath path to be truncated
     * @return base service path
     */
    private static String truncatePath(final String originalPath) {
        String noTrailingSlashPath = removeEnd(originalPath, "/");

        // handle /services[/SecurityServer][/]
        String noCrowdServicePath = removeEnd(noTrailingSlashPath, "/" + CROWD_SERVICE_LOCATION);
        String noSecurityServerPath = removeEnd(noTrailingSlashPath, "/" + CROWD_SERVICE_LOCATION + "/" + SECURITY_SERVER_NAME);

        // shorter path is truncated one => return it
        if (noCrowdServicePath.length() < noSecurityServerPath.length()) {
            return noCrowdServicePath;
        } else {
            return noSecurityServerPath;
        }
    }

    public static ClientPropertiesImpl newInstanceFromResourceLocator(ResourceLocator resourceLocator) {
        Properties properties = resourceLocator.getProperties();
        return newInstanceFromProperties(properties);
    }

    public static ClientPropertiesImpl newInstanceFromProperties(Properties properties) {
        ClientPropertiesImpl clientProperties = new ClientPropertiesImpl();
        clientProperties.updateProperties(properties);
        return clientProperties;
    }
}
