package com.atlassian.crowd.integration.http.util;

import java.util.Arrays;
import java.util.Date;
import java.util.List;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.integration.Constants;
import com.atlassian.crowd.model.authentication.CookieConfiguration;
import com.atlassian.crowd.model.authentication.UserAuthenticationContext;
import com.atlassian.crowd.model.authentication.ValidationFactor;
import com.atlassian.crowd.service.client.ClientProperties;
import com.atlassian.security.cookie.HttpOnlyCookies;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

/**
 * Helper class for Crowd SSO token operations.
 */
public class CrowdHttpTokenHelperImpl implements CrowdHttpTokenHelper {
    private final static Logger LOGGER = LoggerFactory.getLogger(CrowdHttpTokenHelperImpl.class);

    private final CrowdHttpValidationFactorExtractor validationFactorExtractor;

    private CrowdHttpTokenHelperImpl(CrowdHttpValidationFactorExtractor validationFactorExtractor) {
        this.validationFactorExtractor = validationFactorExtractor;
    }

    public String getCrowdToken(HttpServletRequest request, String tokenName) {
        Validate.notNull(request);
        Validate.notNull(tokenName);

        // NOTE: we don't look in the session anymore as we are going to rely on
        // the user supplied cookie (as other authentications may change the cookie
        // and the cookie and session may get out of sync).

        // check request object for token
        // this will exist if an authenticate call is made on the same request as an isAuthenticated call

        LOGGER.debug("Checking for a SSO token that will need to be verified by Crowd.");

        String token = (String) request.getAttribute(Constants.COOKIE_TOKEN_KEY);

        // if there is no token in the request attributes, check cookies
        if (token == null) {
            LOGGER.debug("No request attribute token could be found, now checking the browser submitted cookies.");

            // check the cookies
            Cookie cookies[] = request.getCookies();
            if (cookies != null) {
                if (LOGGER.isDebugEnabled()) {
                    for (Cookie cookie : cookies) {
                        LOGGER.debug("Cookie name/value: " + cookie.getName() + " / " + cookie.getValue());
                    }
                }

                Iterable<Cookie> tokenCookies = Iterables.filter(Arrays.asList(cookies), nonEmptyCookiesCalled(tokenName));

                if (!Iterables.isEmpty(tokenCookies)) {
                    token = tokenCookies.iterator().next().getValue();

                    LOGGER.debug("Accepting the SSO cookie value: {}", token);
                }
            }
        }

        if (LOGGER.isDebugEnabled()) {
            if (token == null) {
                LOGGER.debug("Unable to find a valid Crowd token.");
            } else {
                LOGGER.debug("Existing token value yet to be verified by Crowd: " + token);
            }
        }

        return token;
    }

    public void removeCrowdToken(HttpServletRequest request, HttpServletResponse response, ClientProperties clientProperties, CookieConfiguration cookieConfig) {
        Validate.notNull(request);
        Validate.notNull(clientProperties);
        if (response != null) {
            Validate.notNull(cookieConfig);
        }

        HttpSession session = request.getSession();

        session.removeAttribute(clientProperties.getSessionTokenKey());

        // Remove the token from the request attribute if it exists
        request.removeAttribute(Constants.COOKIE_TOKEN_KEY);

        // set the client cookie flags
        // A zero value causes the cookie to be deleted.
        // fix for Confluence where the response filter is sometimes null.
        if (response != null) {
            Cookie tokenCookie = buildCookie(null, clientProperties.getCookieTokenKey(cookieConfig.getName()), cookieConfig, clientProperties);

            tokenCookie.setMaxAge(0);
            HttpOnlyCookies.addHttpOnlyCookie(response, tokenCookie);
        }
    }

    public void setCrowdToken(HttpServletRequest request, HttpServletResponse response, String token, ClientProperties clientProperties, CookieConfiguration cookieConfig) {
        Validate.notNull(request);
        Validate.notNull(token);
        Validate.notNull(clientProperties);
        if (response != null) {
            Validate.notNull(cookieConfig);
        }
        HttpSession session = request.getSession();

        session.setAttribute(clientProperties.getSessionLastValidation(), new Date());

        // Set the Token on the request, so we know we are authenticated for the life of this request
        request.setAttribute(Constants.COOKIE_TOKEN_KEY, token);

        // fix for Confluence where the response filter is sometimes null.
        if (response != null && request.getAttribute(Constants.REQUEST_SSO_COOKIE_COMMITTED) == null) {
            // create the cookie sent to the client
            Cookie tokenCookie = buildCookie(token, clientProperties.getCookieTokenKey(cookieConfig.getName()), cookieConfig,
                    clientProperties);


            HttpOnlyCookies.addHttpOnlyCookie(response, tokenCookie);
            request.setAttribute(Constants.REQUEST_SSO_COOKIE_COMMITTED, Boolean.TRUE);
        }
    }

    public UserAuthenticationContext getUserAuthenticationContext(HttpServletRequest request, String username, String password, ClientProperties clientProperties) {
        PasswordCredential credential = new PasswordCredential(password);

        UserAuthenticationContext userAuthenticationContext = new UserAuthenticationContext();

        userAuthenticationContext.setApplication(clientProperties.getApplicationName());
        userAuthenticationContext.setCredential(credential);
        userAuthenticationContext.setName(username);
        List<ValidationFactor> validationFactors = validationFactorExtractor.getValidationFactors(request);
        userAuthenticationContext.setValidationFactors(validationFactors.toArray(new ValidationFactor[0]));

        return userAuthenticationContext;
    }

    public CrowdHttpValidationFactorExtractor getValidationFactorExtractor() {
        return validationFactorExtractor;
    }

    /**
     * Creates the cookie and sets attributes such as path, domain, and "secure" flag.
     *
     * @param token            The SSO token to be included in the cookie
     * @param tokenCookieKey   Cookie key for the token
     * @param cookieConfig     Cookie configuration
     * @param clientProperties Client properties
     * @return new cookie
     */
    @SuppressFBWarnings(value = "INSECURE_COOKIE", justification = "setSecure() called if configured")
    private Cookie buildCookie(final String token,
                               final String tokenCookieKey,
                               final CookieConfiguration cookieConfig,
                               final ClientProperties clientProperties) {
        String domain;
        if (!StringUtils.isBlank(clientProperties.getSSOCookieDomainName())) {
            domain = clientProperties.getSSOCookieDomainName();
        } else {
            domain = cookieConfig.getDomain();
        }
        boolean isSecure = cookieConfig.isSecure();
        Cookie tokenCookie = new Cookie(tokenCookieKey, token);

        // path
        tokenCookie.setPath(Constants.COOKIE_PATH);

        // domain
        if (domain != null && StringUtils.isNotBlank(domain) && !"localhost".equals(domain)) {
            tokenCookie.setDomain(domain);
        }

        // "Secure" flag
        tokenCookie.setSecure(isSecure);

        return tokenCookie;
    }

    /**
     * Returns an instance of CrowdHttpTokenHelper.
     *
     * @return CrowdHttpTokenHelper.
     */
    public static CrowdHttpTokenHelper getInstance(final CrowdHttpValidationFactorExtractor validationFactorExtractor) {
        return new CrowdHttpTokenHelperImpl(validationFactorExtractor);
    }

    private static Predicate<Cookie> nonEmptyCookiesCalled(final String tokenName) {
        return new Predicate<Cookie>() {
            @Override
            public boolean apply(Cookie cookie) {
                return tokenName.equals(cookie.getName()) && cookie.getValue() != null;
            }
        };
    }
}
