package com.atlassian.crowd.model.group;

import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.SetMultimap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Date;
import java.util.Objects;
import java.util.Set;

public final class ImmutableDirectoryGroupWithAttributes extends BaseImmutableGroup implements GroupWithAttributes, InternalDirectoryGroup {
    private final ImmutableSetMultimap<String, String> attributes;
    private final Date createdDate;
    private final Date updatedDate;
    private final boolean isLocal;

    private ImmutableDirectoryGroupWithAttributes(Builder builder) {
        super(builder);
        this.attributes = builder.attributes;
        this.createdDate = builder.createdDate;
        this.updatedDate = builder.updatedDate;
        this.isLocal = builder.isLocal;
    }

    public static Builder builder(InternalDirectoryGroup group, SetMultimap<String, String> attributes) {
        return new Builder(group, attributes);
    }

    @Nonnull
    @Override
    public Set<String> getValues(String key) {
        return attributes.get(key);
    }

    @Nullable
    @Override
    public String getValue(String key) {
        return Iterables.getFirst(getValues(key), null);
    }

    @Override
    public Set<String> getKeys() {
        return attributes.keySet();
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    @Override
    public Date getCreatedDate() {
        return createdDate == null ? null : new Date(createdDate.getTime());
    }

    @Override
    public Date getUpdatedDate() {
        return updatedDate == null ? null : new Date(updatedDate.getTime());
    }

    @Override
    public boolean isLocal() {
        return isLocal;
    }

    @Override
    public ImmutableDirectoryGroupWithAttributes withName(String name) {
        return builder(this, this.attributes).setName(name).build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        ImmutableDirectoryGroupWithAttributes that = (ImmutableDirectoryGroupWithAttributes) o;
        return isLocal == that.isLocal &&
                Objects.equals(attributes, that.attributes) &&
                Objects.equals(createdDate, that.createdDate) &&
                Objects.equals(updatedDate, that.updatedDate);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), attributes, createdDate, updatedDate, isLocal);
    }

    public static class Builder extends BaseImmutableGroup.Builder<Builder> {
        private ImmutableSetMultimap<String, String> attributes;
        private Date createdDate;
        private Date updatedDate;
        private boolean isLocal;

        public Builder(InternalDirectoryGroup group, SetMultimap<String, String> attributes) {
            super(group);
            setAttributes(attributes);
            setCreatedDate(group.getCreatedDate());
            setUpdatedDate(group.getUpdatedDate());
            setLocal(group.isLocal());
        }

        public Builder setAttributes(SetMultimap<String, String> attributes) {
            this.attributes = ImmutableSetMultimap.copyOf(attributes);
            return this;
        }

        public Builder setCreatedDate(Date createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public Builder setUpdatedDate(Date updatedDate) {
            this.updatedDate = updatedDate;
            return this;
        }

        public Builder setLocal(boolean local) {
            isLocal = local;
            return this;
        }
        @Override
        public ImmutableDirectoryGroupWithAttributes build() {
            return new ImmutableDirectoryGroupWithAttributes(this);
        }
    }
}
